package ru.yandex.solomon.metrics.client.combined;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.metrics.client.AbstractRequest;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class FindAndReadManyRequest extends AbstractRequest {
    private final Selectors selectors;
    private final boolean useNewFormat;
    private final int metabaseLimit;
    private final List<Operation> operations;
    private final long fromMillis;
    private final long toMillis;
    private final long softReadDeadline;

    private final boolean isOldMode;
    private final DataRequestType requestType;

    private FindAndReadManyRequest(Builder builder) {
        super(builder);
        this.selectors = Objects.requireNonNull(builder.selectors, "selectors");
        this.useNewFormat = builder.useNewFormat;
        this.metabaseLimit = builder.metabaseLimit;
        this.operations = List.copyOf(builder.operations);
        this.fromMillis = requireGoodMillis(builder.fromMillis, "fromMillis");
        this.toMillis = requireGoodMillis(builder.toMillis, "toMillis");
        this.softReadDeadline = builder.softReadDeadline;

        this.isOldMode = builder.isOldMode;
        this.requestType = builder.requestType;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Selectors getSelectors() {
        return selectors;
    }

    public boolean isUseNewFormat() {
        return useNewFormat;
    }

    public int getMetabaseLimit() {
        return metabaseLimit;
    }

    public List<Operation> getOperations() {
        return operations;
    }

    public long getFromMillis() {
        return fromMillis;
    }

    public long getToMillis() {
        return toMillis;
    }

    public long getSoftReadDeadline() {
        return softReadDeadline;
    }

    public boolean isOldMode() {
        return isOldMode;
    }

    public DataRequestType getRequestType() {
        return requestType;
    }

    private static long requireGoodMillis(long millis, String field) {
        if (InstantUtils.isGoodMillis(millis)) {
            return millis;
        }

        throw new IllegalArgumentException("Bad number for " + field + ", got: " + millis);
    }

    public static class Builder extends AbstractRequest.Builder<Builder> {
        private Selectors selectors = null;
        private boolean useNewFormat = false;
        private int metabaseLimit = Integer.MAX_VALUE;
        private final List<Operation> operations = new ArrayList<>();
        private long fromMillis;
        private long toMillis;
        private long softReadDeadline;

        private boolean isOldMode;
        private DataRequestType requestType;

        public Builder setSelectors(Selectors selectors) {
            this.selectors = selectors;
            return this;
        }

        public Selectors getSelectors() {
            return selectors;
        }

        public Builder setUseNewFormat(boolean useNewFormat) {
            this.useNewFormat = useNewFormat;
            return this;
        }

        public Builder setMetabaseLimit(int metabaseLimit) {
            this.metabaseLimit = metabaseLimit;
            return this;
        }

        public Builder setOperations(List<Operation> operations) {
            this.operations.clear();
            this.operations.addAll(operations);
            return this;
        }

        public Builder addOperation(Operation operation) {
            this.operations.add(operation);
            return this;
        }

        public Builder setFromMillis(long fromMillis) {
            this.fromMillis = fromMillis;
            return this;
        }

        public Builder setToMillis(long toMillis) {
            this.toMillis = toMillis;
            return this;
        }

        public Builder setSoftReadDeadline(long softReadDeadline) {
            this.softReadDeadline = softReadDeadline;
            return this;
        }

        public Builder setFrom(Instant from) {
            return setFromMillis(from.toEpochMilli());
        }

        public Builder setTo(Instant to) {
            return setToMillis(to.toEpochMilli());
        }

        public Builder setSoftReadDeadline(Instant softReadDeadline) {
            return setSoftReadDeadline(softReadDeadline.toEpochMilli());
        }

        public Builder setOldMode(boolean oldMode) {
            isOldMode = oldMode;
            return this;
        }

        public Builder setRequestType(DataRequestType requestType) {
            this.requestType = requestType;
            return this;
        }

        public FindAndReadManyRequest build() {
            return new FindAndReadManyRequest(this);
        }
    }

    @Override
    public String toString() {
        return "FindAndReadManyRequest{" +
                "selectors=" + selectors +
                ", useNewFormat=" + useNewFormat +
                ", metabaseLimit=" + metabaseLimit +
                ", operations=" + operations +
                ", fromMillis=" + fromMillis +
                ", toMillis=" + toMillis +
                ", softReadDeadline=" + softReadDeadline +
                ", isOldMode=" + isOldMode +
                ", requestType=" + requestType +
                '}';
    }
}
