package ru.yandex.solomon.metrics.client.combined;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationAggregationSummary;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.math.protobuf.OperationTop;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.model.timeseries.aggregation.Int64Summary;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
public class TopOptimization {
    private static final int MIN_METRICS_FOR_RANK_OPTIMIZATION = 200;

    public static CompletableFuture<ReadManyResponse> readMany(MetricsClient client, ReadManyRequest request) {
        if (request.getKeys().size() < MIN_METRICS_FOR_RANK_OPTIMIZATION) {
            return client.readMany(request);
        }

        var operations = request.getOperations();
        int opTopIdx = topOperationIdx(operations);
        if (opTopIdx == -1) {
            return client.readMany(request);
        }

        var opTop = operations.get(opTopIdx).getTop();
        if (opTop.getLimit() >= request.getKeys().size()) {
            return client.readMany(request);
        }

        return client.readMany(request.toBuilder()
                .setOperations(operations.subList(0, opTopIdx))
                .addOperation(opSummary(opTop.getTimeAggregation()))
                .addOperation(opDropTimeseries())
                .build())
                .thenCompose(response -> {
                    if (!response.isOk()) {
                        return CompletableFuture.completedFuture(response);
                    }

                    var filtered = applyTop(opTop, response.getMetrics());
                    return client.readMany(overrideRequest(request, filtered, opTopIdx));
                });
    }

    private static ReadManyRequest overrideRequest(ReadManyRequest request, List<MetricKey> filtered, int topIdx) {
        var builder = request.toBuilder();
        builder.setKeys(filtered);
        var operations = new ArrayList<>(request.getOperations());
        operations.remove(topIdx);
        builder.setOperations(operations);
        return builder.build();
    }

    private static List<MetricKey> applyTop(OperationTop op, List<Metric<MetricKey>> metrics) {
        return metrics.stream()
                .map(metric -> Rank.of(metric, op))
                .filter(Objects::nonNull)
                .sorted(rankCompare(op))
                .limit(op.getLimit())
                .map(Rank::key)
                .collect(Collectors.toList());
    }

    private static Comparator<? super Rank> rankCompare(OperationTop op) {
        Comparator<Rank> comparator = Comparator.comparing(Rank::rank);
        if (!op.getAsc()) {
            comparator = comparator.reversed();
        }

        return Comparator.comparing(Rank::isEmpty)
                .thenComparing(comparator);
    }

    private static Double extractAggregation(TimeseriesSummary summary, Aggregation aggr) {
        if (summary instanceof DoubleSummary) {
            return extractDoubleAggregation((DoubleSummary) summary, aggr);
        }
        if (summary instanceof Int64Summary) {
            return Nullables.map(extractLongAggregation((Int64Summary) summary, aggr), Long::doubleValue);
        }
        return null;
    }

    private static Long extractLongAggregation(Int64Summary summary, Aggregation aggr) {
        switch (aggr) {
            case MAX:
                return summary.getMax();
            case MIN:
                return summary.getMin();
            case SUM:
                return summary.getSum();
            case AVG:
                return summary.getAvg();
            case LAST:
                return summary.getLast();
            case COUNT:
                return summary.getCount();
            default:
                return null;
        }
    }

    private static Double extractDoubleAggregation(DoubleSummary summary, Aggregation aggr) {
        switch (aggr) {
            case MAX:
                return summary.getMax();
            case MIN:
                return summary.getMin();
            case SUM:
                return summary.getSum();
            case AVG:
                return summary.getAvg();
            case LAST:
                return summary.getLast();
            case COUNT:
                return (double) summary.getCount();
            default:
                return null;
        }
    }

    private static Operation opSummary(Aggregation aggregation) {
        var op = OperationAggregationSummary.newBuilder().addAggregations(aggregation).build();
        return Operation.newBuilder().setSummary(op).build();
    }

    private static Operation opDropTimeseries() {
        var op = OperationDropTimeSeries.newBuilder().build();
        return Operation.newBuilder().setDropTimeseries(op).build();
    }

    private static int topOperationIdx(List<Operation> operations) {
        for (int index = 0; index < operations.size(); index++) {
            var op = operations.get(index);
            if (op.hasTop()) {
                return index;
            }
        }
        return -1;
    }

    private static class Rank {
        private final MetricKey key;
        private final double rank;
        private final boolean empty;

        private Rank(MetricKey key, double rank, boolean empty) {
            this.key = key;
            this.rank = rank;
            this.empty = empty;
        }

        @Nullable
        public static Rank of(Metric<MetricKey> metric, OperationTop op) {
            var summary = metric.getSummary();
            if (summary == null || !summary.has(op.getTimeAggregation())) {
                return null;
            }

            var rank = extractAggregation(summary, op.getTimeAggregation());
            if (rank == null) {
                return null;
            }

            return new Rank(metric.getKey(), rank, summary.getCount() == 0);
        }

        public MetricKey key() {
            return key;
        }

        public double rank() {
            return rank;
        }

        public boolean isEmpty() {
            return empty;
        }
    }
}
