package ru.yandex.solomon.metrics.client.dataproxy;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.dataproxy.client.DataProxyClient;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metrics.client.AbstractRequest;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.FindResponse;
import ru.yandex.solomon.metrics.client.LabelNamesRequest;
import ru.yandex.solomon.metrics.client.LabelNamesResponse;
import ru.yandex.solomon.metrics.client.LabelValuesRequest;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.metrics.client.MetricNamesRequest;
import ru.yandex.solomon.metrics.client.MetricNamesResponse;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.MetricsClientMetrics;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.metrics.client.ReadRequest;
import ru.yandex.solomon.metrics.client.ReadResponse;
import ru.yandex.solomon.metrics.client.ResolveManyRequest;
import ru.yandex.solomon.metrics.client.ResolveManyResponse;
import ru.yandex.solomon.metrics.client.ResolveManyWithNameRequest;
import ru.yandex.solomon.metrics.client.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.ResolveOneResponse;
import ru.yandex.solomon.metrics.client.ResolveOneWithNameRequest;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.metrics.client.UniqueLabelsRequest;
import ru.yandex.solomon.metrics.client.UniqueLabelsResponse;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.stockpile.api.EStockpileStatusCode;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class DataProxyAwareClient implements MetricsClient {

    private static final Logger logger = LoggerFactory.getLogger(DataProxyAwareClient.class);

    private final DataProxyClient client;
    private final MetricsClient delegate;
    private final FeatureFlagsHolder flagsHolder;
    private final MetricsClientMetrics metricsClientMetrics;

    private final FeatureFlag metaFlag;
    private final FeatureFlag dataFlag;

    public DataProxyAwareClient(
            @WillCloseWhenClosed DataProxyClient client,
            @WillCloseWhenClosed MetricsClient delegate,
            String clientId,
            FeatureFlagsHolder flagsHolder,
            @Nullable MetricsClientMetrics metricsClientMetrics)
    {
        this.client = client;
        this.delegate = delegate;
        this.flagsHolder = flagsHolder;
        this.metricsClientMetrics = metricsClientMetrics;

        // Allows separate feature flags control over alerting and gateway
        if (clientId.startsWith("solomon-alerting-")) {
            metaFlag = FeatureFlag.ALERTING_DATAPROXY;
            dataFlag = FeatureFlag.ALERTING_DATAPROXY;
        } else {
            metaFlag = FeatureFlag.METADATA_FROM_DATAPROXY;
            dataFlag = FeatureFlag.SERIES_FROM_DATAPROXY;
        }
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        return dispatchBySelectors(request, FindRequest::getSelectors, "find", metaFlag,
                delegate::find,
                projectId -> client.find(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new FindResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        return dispatchByLabels(request, ResolveOneRequest::getLabels, "resolveOne", metaFlag,
                delegate::resolveOne,
                projectId -> client.resolveOne(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new ResolveOneResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOneWithName(ResolveOneWithNameRequest request) {
        return dispatchByLabels(request, r -> r.getMetric().getLabels(), "resolveOneWithName", metaFlag,
                delegate::resolveOneWithName,
                projectId -> client.resolveOne(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new ResolveOneResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        return dispatchByLabels(request, ResolveManyRequest::getCommonLabels, "resolveMany", metaFlag,
                delegate::resolveMany,
                projectId -> client.resolveMany(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new ResolveManyResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveManyWithName(ResolveManyWithNameRequest request) {
        return dispatchByLabels(request, ResolveManyWithNameRequest::getCommonLabels, "resolveManyWithName", metaFlag,
                delegate::resolveManyWithName,
                projectId -> client.resolveMany(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new ResolveManyResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        return dispatchBySelectors(request, MetricNamesRequest::getSelectors, "metricNames", metaFlag,
                delegate::metricNames,
                projectId -> client.metricNames(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new MetricNamesResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<LabelNamesResponse> labelNames(LabelNamesRequest request) {
        return dispatchBySelectors(request, LabelNamesRequest::getSelectors, "labelNames", metaFlag,
                delegate::labelNames,
                projectId -> client.labelKeys(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new LabelNamesResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<LabelValuesResponse> labelValues(LabelValuesRequest request) {
        return dispatchBySelectors(request, LabelValuesRequest::getSelectors, "labelValues", metaFlag,
                delegate::labelValues,
                projectId -> client.labelValues(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new LabelValuesResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp, delegate.getDestinations());
                        }));
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request) {
        return dispatchBySelectors(request, UniqueLabelsRequest::getSelectors, "uniqueLabels", metaFlag,
                delegate::uniqueLabels,
                projectId -> client.uniqueLabels(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new UniqueLabelsResponse(classifyError(throwable));
                            }
                            return Glue.pasteUp(resp);
                        }));
    }

    @Override
    public CompletableFuture<ReadResponse> read(ReadRequest request) {
        return delegate.read(request);
    }

    @Override
    public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request) {
        return delegate.readMany(request);
    }

    @Override
    public AvailabilityStatus getMetabaseAvailability() {
        return delegate.getMetabaseAvailability();
    }

    @Override
    public AvailabilityStatus getStockpileAvailability() {
        return delegate.getStockpileAvailability();
    }

    @Override
    public Stream<Labels> metabaseShards(String destination, Selectors selector) {
        return delegate.metabaseShards(destination, selector);
    }

    @Override
    public String getStockpileHostForShardId(String destination, int shardId) {
        return delegate.getStockpileHostForShardId(destination, shardId);
    }

    @Override
    public CompletableFuture<FindAndReadManyResponse> findAndReadMany(FindAndReadManyRequest request) {
        return dispatchBySelectors(request, FindAndReadManyRequest::getSelectors, "findAndReadMany", dataFlag,
                req -> delegate.findAndReadMany(req)
                        .thenApply(readResponse -> {
                            if (metricsClientMetrics == null) {
                                return readResponse;
                            }
                            for (Metric<MetricKey> metric : readResponse.getMetrics()) {
                                if (metric.getTimeseries() != null) {
                                    metricsClientMetrics.hitMetricsRead(request, metric.getTimeseries().elapsedBytes(), metric.getTimeseries().getRecordCount());
                                }
                            }
                            return readResponse;
                        }),
                projectId -> client.readMany(Glue.pasteUp(projectId, request), request.getDeadline())
                        .handle((resp, throwable) -> {
                            if (throwable != null) {
                                return new FindAndReadManyResponse(classifyDataError(throwable));
                            }
                            var result = Glue.pasteUp(resp);
                            if (metricsClientMetrics == null) {
                                return result;
                            }
                            for (Metric<MetricKey> metric : result.getMetrics()) {
                                if (metric.getTimeseries() != null) {
                                    metricsClientMetrics.hitMetricsRead(request, metric.getTimeseries().elapsedBytes(), metric.getTimeseries().getRecordCount());
                                }
                            }
                            return result;
                        }));
    }

    @Override
    public Collection<String> getDestinations() {
        return delegate.getDestinations();
    }

    @Override
    public void close() {
        client.close();
        delegate.close();
    }

    private static MetabaseStatus classifyError(Throwable throwable) {
        var cause = CompletableFutures.unwrapCompletionException(throwable);
        if (cause instanceof StatusRuntimeException) {
            Status status = ((StatusRuntimeException) cause).getStatus();
            switch (status.getCode()) {
                case NOT_FOUND:
                    return MetabaseStatus.fromCode(EMetabaseStatusCode.SHARD_NOT_FOUND, status.getDescription());
                case DEADLINE_EXCEEDED:
                    return MetabaseStatus.fromCode(EMetabaseStatusCode.DEADLINE_EXCEEDED, status.getDescription());
                case FAILED_PRECONDITION:
                    return MetabaseStatus.fromCode(EMetabaseStatusCode.INVALID_REQUEST, status.getDescription());
                case UNAVAILABLE:
                    return MetabaseStatus.fromCode(EMetabaseStatusCode.NODE_UNAVAILABLE, status.getDescription());
            }
        }

        return MetabaseStatus.fromThrowable(cause);
    }

    private static StockpileStatus classifyDataError(Throwable throwable) {
        var cause = CompletableFutures.unwrapCompletionException(throwable);
        if (cause instanceof StatusRuntimeException) {
            Status status = ((StatusRuntimeException) cause).getStatus();
            switch (status.getCode()) {
                case NOT_FOUND:
                    return StockpileStatus.fromCode(EStockpileStatusCode.METRIC_NOT_FOUND, status.getDescription());
                case DEADLINE_EXCEEDED:
                    return StockpileStatus.fromCode(EStockpileStatusCode.DEADLINE_EXCEEDED, status.getDescription());
                case FAILED_PRECONDITION:
                    return StockpileStatus.fromCode(EStockpileStatusCode.INVALID_REQUEST, status.getDescription());
                case UNAVAILABLE:
                    return StockpileStatus.fromCode(EStockpileStatusCode.NODE_UNAVAILABLE, status.getDescription());
            }
        }

        return StockpileStatus.fromThrowable(cause);
    }

    private <ReqT extends AbstractRequest, RespT> RespT dispatchBySelectors(
            ReqT request,
            Function<ReqT, Selectors> extractor,
            String methodStr,
            FeatureFlag flag,
            Function<ReqT, RespT> delegateMethod,
            Function<String, RespT> clientMethod)
    {
        if (!request.isDataProxyAllowed()) {
            return delegateMethod.apply(request);
        }

        Selectors selectors = extractor.apply(request);
        Selector projectSelector = selectors.findByKey(LabelKeys.PROJECT);
        if (projectSelector == null || !projectSelector.isExact()) {
            logger.warn("trying to call " + methodStr + "() with not exact project selector {}", selectors);
            return delegateMethod.apply(request);
        }

        String projectId = projectSelector.getValue();
        if (!flagsHolder.hasFlag(flag, projectId)) {
            return delegateMethod.apply(request);
        }

        return clientMethod.apply(projectId);
    }

    private <ReqT, RespT> RespT dispatchByLabels(
            ReqT request,
            Function<ReqT, Labels> extractor,
            String methodStr,
            FeatureFlag flag,
            Function<ReqT, RespT> delegateMethod,
            Function<String, RespT> clientMethod)
    {
        Labels labels = extractor.apply(request);
        Label projectLabel = labels.findByKey("project");
        if (projectLabel == null) {
            logger.warn("trying to call " + methodStr + "() without project label {}", labels);
            return delegateMethod.apply(request);
        }

        String projectId = projectLabel.getValue();
        if (!flagsHolder.hasFlag(flag, projectId)) {
            return delegateMethod.apply(request);
        }

        return clientMethod.apply(projectId);
    }
}
