package ru.yandex.solomon.metrics.client.dataproxy;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Maps;

import ru.yandex.monitoring.dataproxy.LabelKeysRequest;
import ru.yandex.monitoring.dataproxy.LabelValuesResponse.LabelValues;
import ru.yandex.monitoring.dataproxy.MetricPooled;
import ru.yandex.monitoring.dataproxy.ReadManyRequest;
import ru.yandex.monitoring.dataproxy.ReadManyResponse;
import ru.yandex.monitoring.dataproxy.UniqueLabelsResponse.LabelsPooled;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.common.StringPool.Compression;
import ru.yandex.solomon.dataproxy.client.LabelsConverter;
import ru.yandex.solomon.dataproxy.client.MetricConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.FindResponse;
import ru.yandex.solomon.metrics.client.LabelNamesRequest;
import ru.yandex.solomon.metrics.client.LabelNamesResponse;
import ru.yandex.solomon.metrics.client.LabelValuesRequest;
import ru.yandex.solomon.metrics.client.LabelValuesResponse;
import ru.yandex.solomon.metrics.client.MetricNamesRequest;
import ru.yandex.solomon.metrics.client.MetricNamesResponse;
import ru.yandex.solomon.metrics.client.ResolveManyRequest;
import ru.yandex.solomon.metrics.client.ResolveManyResponse;
import ru.yandex.solomon.metrics.client.ResolveManyWithNameRequest;
import ru.yandex.solomon.metrics.client.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.ResolveOneResponse;
import ru.yandex.solomon.metrics.client.ResolveOneWithNameRequest;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.metrics.client.UniqueLabelsRequest;
import ru.yandex.solomon.metrics.client.UniqueLabelsResponse;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.metrics.client.combined.OldModeResult;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.solomon.util.labelStats.LabelValuesStats;
import ru.yandex.solomon.util.protobuf.StringPool;

/**
 * @author Sergey Polovko
 */
final class Glue {
    private Glue() {}

    // Find

    public static ru.yandex.monitoring.dataproxy.FindRequest pasteUp(String project, FindRequest req) {
        var builder = ru.yandex.monitoring.dataproxy.FindRequest.newBuilder()
                .setSelectors(req.getSelectors().toString())
                .setProjectId(project)
                .setLimit(req.getLimit())
                .setFillMetricName(req.isUseNewFormat());
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static FindResponse pasteUp(ru.yandex.monitoring.dataproxy.FindResponse resp) {
        Map<String, Integer> metricsCountByReplica = Maps.newHashMapWithExpectedSize(2);
        if (!resp.getReplicaName1().isEmpty()) {
            metricsCountByReplica.put(resp.getReplicaName1(), resp.getTotalCount());
        }
        if (!resp.getReplicaName2().isEmpty()) {
            metricsCountByReplica.put(resp.getReplicaName2(), resp.getTotalCount());
        }
        List<MetricKey> keys = MetricConverter.toMetricKeys(resp);
        return new FindResponse(keys, resp.getTotalCount(), metricsCountByReplica, resp.getTruncated(), true);
    }

    // ResolveOne + ResolveOneWithName

    public static ru.yandex.monitoring.dataproxy.ResolveOneRequest pasteUp(String project, ResolveOneRequest req) {
        var builder = ru.yandex.monitoring.dataproxy.ResolveOneRequest.newBuilder()
                .setProjectId(project)
                .addAllLabels(LabelsConverter.toStringArray(req.getLabels()));
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static ru.yandex.monitoring.dataproxy.ResolveOneRequest pasteUp(String project, ResolveOneWithNameRequest req) {
        return ru.yandex.monitoring.dataproxy.ResolveOneRequest.newBuilder()
                .setProjectId(project)
                .setName(req.getMetric().getName())
                .addAllLabels(LabelsConverter.toStringArray(req.getMetric().getLabels()))
                .build();
    }

    public static ResolveOneResponse pasteUp(ru.yandex.monitoring.dataproxy.ResolveOneResponse resp) {
        return new ResolveOneResponse(MetricConverter.toMetricKey(resp));
    }

    // ResolveMany + ResolveManyWithName

    public static ru.yandex.monitoring.dataproxy.ResolveManyRequest pasteUp(String projectId, ResolveManyRequest req) {
        var stringsBuilder = StringPool.newBuilder();
        var commonLabels = LabelsConverter.toIntArray(stringsBuilder, req.getCommonLabels());
        var metricKeys = MetricConverter.fromLabelsList(stringsBuilder, req.getLabelsList());
        var strings = stringsBuilder.buildProto(Compression.LZ4);

        var builder = ru.yandex.monitoring.dataproxy.ResolveManyRequest.newBuilder()
                .setProjectId(projectId)
                .setStringPool(strings)
                .addAllCommonLabelsIdx(commonLabels)
                .addAllMetricKeys(metricKeys);
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static ru.yandex.monitoring.dataproxy.ResolveManyRequest pasteUp(String projectId, ResolveManyWithNameRequest req) {
        var stringsBuilder = StringPool.newBuilder();
        var commonLabels = LabelsConverter.toIntArray(stringsBuilder, req.getCommonLabels());
        var metricKeys = MetricConverter.fromMetrics(stringsBuilder, req.getMetrics());
        var strings = stringsBuilder.buildProto(Compression.LZ4);

        var builder = ru.yandex.monitoring.dataproxy.ResolveManyRequest.newBuilder()
                .setProjectId(projectId)
                .setStringPool(strings)
                .addAllCommonLabelsIdx(commonLabels)
                .addAllMetricKeys(metricKeys);
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static ResolveManyResponse pasteUp(ru.yandex.monitoring.dataproxy.ResolveManyResponse resp) {
        return new ResolveManyResponse(MetricConverter.toMetricKeys(resp));
    }

    // MetricNames

    public static ru.yandex.monitoring.dataproxy.MetricNamesRequest pasteUp(String projectId, MetricNamesRequest req) {
        var builder = ru.yandex.monitoring.dataproxy.MetricNamesRequest.newBuilder()
                .setProjectId(projectId)
                .setLimit(req.getLimit())
                .setSelectors(req.getSelectors().toString())
                .setTextFilter(req.getTextSearch());
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static MetricNamesResponse pasteUp(ru.yandex.monitoring.dataproxy.MetricNamesResponse resp) {
        int size = resp.getNamesIdxCount();
        var names = new ArrayList<String>(size);

        var strings = StringPool.fromProto(resp.getStringPool());
        for (int i = 0; i < size; i++) {
            names.add(strings.get(resp.getNamesIdx(i)));
        }
        return new MetricNamesResponse(names, resp.getTruncated());
    }

    // LabelKeys

    public static ru.yandex.monitoring.dataproxy.LabelKeysRequest pasteUp(String projectId, LabelNamesRequest req) {
        var builder = LabelKeysRequest.newBuilder()
                .setProjectId(projectId)
                .setSelectors(req.getSelectors().toString());
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static LabelNamesResponse pasteUp(ru.yandex.monitoring.dataproxy.LabelKeysResponse resp) {
        int size = resp.getKeysIdxCount();
        var keys = new HashSet<String>(size);

        var strings = StringPool.fromProto(resp.getStringPool());
        for (int i = 0; i < size; i++) {
            keys.add(strings.get(resp.getKeysIdx(i)));
        }
        return new LabelNamesResponse(keys);
    }

    // LabelValues

    public static ru.yandex.monitoring.dataproxy.LabelValuesRequest pasteUp(String projectId, LabelValuesRequest req) {
        var builder = ru.yandex.monitoring.dataproxy.LabelValuesRequest.newBuilder()
                .setProjectId(projectId)
                .setSelectors(req.getSelectors().toString())
                .addAllKeys(req.getLabelNames())
                .setTextFilter(req.getTextSearch())
                .setLimit(req.getLimit());
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static LabelValuesResponse pasteUp(
            ru.yandex.monitoring.dataproxy.LabelValuesResponse resp,
            Collection<String> destinations)
    {
        var strings = StringPool.fromProto(resp.getStringPool());

        Map<String, LabelStats> byKey = new HashMap<>(resp.getLabelsCount());
        for (LabelValues label : resp.getLabelsList()) {
            var labelStats = LabelStats.create();

            for (int i = 0; i < label.getValuesIdxCount(); i++) {
                labelStats.add(strings.get(label.getValuesIdx(i)));
            }

            labelStats.setTruncated(label.getTruncated());
            labelStats.setCount(label.getMetricCount());
            byKey.put(strings.get(label.getKeyIdx()), labelStats);
        }

        var countByDestination = Maps.<String, Integer>newHashMapWithExpectedSize(destinations.size());
        for (String destination : destinations) {
            countByDestination.put(destination, resp.getMetricCount());
        }

        var stats = new LabelValuesStats(byKey, resp.getMetricCount());
        return new LabelValuesResponse(stats, countByDestination);
    }

    // UniqueLabels

    public static ru.yandex.monitoring.dataproxy.UniqueLabelsRequest pasteUp(String projectId, UniqueLabelsRequest req) {
        var builder = ru.yandex.monitoring.dataproxy.UniqueLabelsRequest.newBuilder()
                .setProjectId(projectId)
                .setSelectors(req.getSelectors().toString())
                .addAllKeys(req.getLabels());
        if (req.getDestination() != null) {
            builder.setForceReplicaRead(req.getDestination());
        }
        return builder.build();
    }

    public static UniqueLabelsResponse pasteUp(ru.yandex.monitoring.dataproxy.UniqueLabelsResponse resp) {
        var strings = StringPool.fromProto(resp.getStringPool());

        var labels = new HashSet<Labels>(resp.getLabelsCount());
        var labelsBuilder = Labels.builder(Labels.MAX_LABELS_COUNT);

        for (LabelsPooled labelsPooled : resp.getLabelsList()) {
            labels.add(LabelsConverter.fromIntArray(strings, labelsBuilder, labelsPooled.getLabelsIdxList()));
        }

        return new UniqueLabelsResponse(labels);
    }

    // FindAndReadMany

    public static ReadManyRequest pasteUp(String projectId, FindAndReadManyRequest request) {
        var builder = ReadManyRequest.newBuilder()
                .setProjectId(projectId)
                .setFromMillis(request.getFromMillis())
                .setToMillis(request.getToMillis())
                .setMaxTimeSeriesFormat(TimeseriesDecoder.getCurrentTimeseriesFormat())
                .setLookup(ReadManyRequest.Lookup.newBuilder()
                        .setSelectors(Selectors.format(request.getSelectors()))
                        .setLimit(request.getMetabaseLimit()))
                .addAllOperations(request.getOperations());

        if (request.getDestination() != null) {
            builder.setForceReplicaRead(request.getDestination());
        }

        return builder.build();
    }

    public static FindAndReadManyResponse pasteUp(ReadManyResponse response) {
        StringPool strings = StringPool.fromProto(response.getStringPool());

        List<Metric<MetricKey>> metrics = response.getMetricsList().stream()
                .map(metricData -> {
                    MetricPooled meta = metricData.getMetric();
                    MetricKey key = MetricConverter.toMetricKey(strings, meta);

                    @Nullable AggrGraphDataIterable timeseries = TimeseriesDecoder.fromProto(metricData);
                    @Nullable TimeseriesSummary summary = AggregateDecoder.fromProto(metricData);

                    MetricType type = TimeSeriesCodec.decodeType(metricData.getTimeSeries());
                    return new Metric<>(key, type, timeseries, summary);
                })
                .collect(Collectors.toList());

        return new FindAndReadManyResponse(metrics, OldModeResult.DEFAULT);
    }
}
