package ru.yandex.solomon.metrics.client.dc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.operation.Operations;
import ru.yandex.solomon.math.operation.OperationsPipelineReduced;
import ru.yandex.solomon.math.operation.SplittedOperations;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.aggregation.AggregateConverters;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.client.StockpileClient;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class DcCallReadMany extends ClientCall<ReadManyRequest, ReadManyResponse> {
    private static final Logger logger = LoggerFactory.getLogger(DcCallReadMany.class);

    private final String destination;
    private final StockpileClient stockpile;
    private final Int2ObjectMap<Shard> keysByShard;
    private final SplittedOperations operations;

    private DcCallReadMany(ReadManyRequest request, String destination, StockpileClient stockpile, Int2ObjectMap<Shard> keysByShard) {
        super(request);
        this.destination = destination;
        this.stockpile = stockpile;
        this.keysByShard = keysByShard;
        if (keysByShard.size() == 1) {
            operations = new SplittedOperations(Collections.emptyList(), request.getOperations());
        } else {
            operations = Operations.splitOperations(request.getOperations());
        }
    }

    public static ClientCall<ReadManyRequest, ReadManyResponse> of(String destination, StockpileClient client, ReadManyRequest request) {
        Int2ObjectMap<Shard> keysByShard = new Int2ObjectOpenHashMap<>(request.getKeys().size());
        for (MetricKey key : request.getKeys()) {
            StockpileKey stockpileKey = getStockpileKey(destination, key);
            if (stockpileKey == null) {
                continue;
            }

            Shard shard = keysByShard.computeIfAbsent(stockpileKey.getShardId(), Shard::new);
            shard.add(stockpileKey.getLocalId(), key);
        }

        if (keysByShard.isEmpty()) {
            StockpileStatus status = StockpileStatus.fromCode(EStockpileStatusCode.METRIC_NOT_FOUND, "Not found metrics at " + destination + " at keys "+ request.getKeys());
            return new ClientCallConstant<>(request, new ReadManyResponse(status));
        }

        return new DcCallReadMany(request, destination, client, keysByShard);
    }

    @Nullable
    private static StockpileKey getStockpileKey(String destination, MetricKey metricKey) {
        for (StockpileKey key : metricKey.getStockpileKeys()) {
            if (destination.equals(key.getDestination())) {
                return key;
            }
        }

        return null;
    }

    public CompletableFuture<ReadManyResponse> call() {
        if (keysByShard.size() == 1) {
            return readMany(keysByShard.values().iterator().next());
        }

        if (operations.getClientOps().isEmpty()) {
            return callCrossShardNoOp();
        }

        return callCrossShard();
    }

    private CompletableFuture<ReadManyResponse> callCrossShardNoOp() {
        return keysByShard.values()
                .stream()
                .map(this::readMany)
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(list -> {
                    List<Metric<MetricKey>> metrics = new ArrayList<>();
                    for (ReadManyResponse response : list) {
                        if (!response.isOk()) {
                            return response;
                        }

                        metrics.addAll(response.getMetrics());
                    }

                    return new ReadManyResponse(metrics);
                });
    }

    private CompletableFuture<ReadManyResponse> callCrossShard() {
        return callCrossShardNoOp()
                .thenCompose(response -> {
                    if (!response.isOk()) {
                        return completedFuture(response);
                    }

                    return new OperationsPipelineReduced<>(completedFuture(response.getMetrics()))
                            .apply(getRequest().getInterval(), operations.getClientOps())
                            .collect(Function.identity())
                            .thenApply(ReadManyResponse::new);
                });
    }

    private CompletableFuture<ReadManyResponse> readMany(Shard shard) {
        StockpileFormat format = defineStockpileFormat();
        return stockpile.readCompressedMany(TReadManyRequest.newBuilder()
                .setShardId(shard.shardId)
                .addAllLocalIds(shard.keyByLocalId.keySet())
                .setBinaryVersion(format.getFormat())
                .setFromMillis(getRequest().getFromMillis())
                .setToMillis(getRequest().getToMillis())
                .addAllOperations(operations.getServerOps())
                .setProducer(getRequest().getProducer())
                .build())
                .thenApply(response -> parseReadManyResponse(shard, response));
    }

    private ReadManyResponse parseReadManyResponse(Shard shard, TCompressedReadManyResponse response) {
        if (response.getStatus() != EStockpileStatusCode.OK) {
            StockpileStatus status = StockpileStatus.fromCode(
                    response.getStatus(),
                    "[" + destination + "] " + response.getStatusMessage());

            if (logger.isDebugEnabled()) {
                logger.debug("Stockpile error({}) - {}", destination, status);
            }

            return new ReadManyResponse(status);
        }

        return response.getMetricsList()
                .stream()
                .map(metricData -> {
                    long localId = metricData.getLocalId();
                    MetricKey key = localId != 0
                            ? shard.keyByLocalId.get(localId)
                            : null;

                    AggrGraphDataIterable timeseries = metricData.hasCompressed()
                            ? TimeSeriesCodec.sequenceDecode(metricData.getCompressed())
                            : null;

                    MetricType type = metricData.getType();
                    if (type == MetricType.METRIC_TYPE_UNSPECIFIED && timeseries != null) {
                        type = StockpileColumns.typeByMask(timeseries.columnSetMask());
                    }

                    TimeseriesSummary summary = AggregateConverters.fromProto(metricData);
                    return new Metric<>(key, type, timeseries, summary);
                })
                .collect(collectingAndThen(toList(), ReadManyResponse::new));
    }

    private StockpileFormat defineStockpileFormat() {
        if (stockpile.getCompatibleCompressFormat().isEmpty()) {
            return StockpileFormat.CURRENT;
        }

        return StockpileFormat.byNumberOrCurrent(stockpile.getCompatibleCompressFormat().upperEndpoint());
    }

    private static class Shard {
        private int shardId;
        private Long2ObjectMap<MetricKey> keyByLocalId;

        Shard(int shardId) {
            this.shardId = shardId;
            this.keyByLocalId = new Long2ObjectOpenHashMap<>();
        }

        public void add(long localId, MetricKey key) {
            this.keyByLocalId.put(localId, key);
        }
    }
}
