package ru.yandex.solomon.metrics.client.exceptions;

import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.jsonschema.JsonSchema;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.CaseFormat;


/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class DataClientException extends RuntimeException {
    static final ObjectMapper mapper = new ObjectMapper();
    static final TypeReference<Map<String, Object>> DETAILS_TYPE = new TypeReference<>() {};

    static {
        mapper.disable(
                MapperFeature.AUTO_DETECT_CREATORS,
                MapperFeature.AUTO_DETECT_FIELDS,
                MapperFeature.AUTO_DETECT_GETTERS,
                MapperFeature.AUTO_DETECT_IS_GETTERS
        );
    }

    public String getType() {
        return getClassNameAsUpperCase(getClass());
    }

    public Map<String, Object> getDetails() {
        return mapper.convertValue(this, DETAILS_TYPE);
    }

    public DataClientException(String message) {
        super(message);
    }

    private static String getClassNameAsUpperCase(Class<? extends DataClientException> clazz) {
        return CaseFormat.UPPER_CAMEL.to(CaseFormat.UPPER_UNDERSCORE, clazz.getSimpleName());
    }

    public static class ExceptionSchema {
        private final String type;
        private final ObjectNode schema;

        public ExceptionSchema(String type, JsonSchema schema) {
            this.type = type;
            this.schema = validateSchema(schema.getSchemaNode());
        }

        private ObjectNode validateSchema(ObjectNode schemaNode) {
            return schemaNode;
        }

        public String getType() {
            return type;
        }

        public ObjectNode getSchema() {
            return schema;
        }

        @Override
        public String toString() {
            return "Schema{" +
                    "type='" + type + '\'' +
                    ", schema=" + schema +
                    '}';
        }
    }

    public static ExceptionSchema getExceptionSchema(Class<? extends DataClientException> clazz) {
        String type = getClassNameAsUpperCase(clazz);
        // TODO: migrate to jackson-module-jsonSchema module (requires matched jackson version)
        try {
            return new ExceptionSchema(type, mapper.generateJsonSchema(clazz));
        } catch (JsonMappingException e) {
            throw new RuntimeException(e);
        }
    }
}
