package ru.yandex.solomon.metrics.client;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.salmon.proto.StockpileCanonicalProto;
import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindRequest;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindResponse;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TMetabaseStatus;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TMetricKey;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadRequest;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadResponse;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TStockpileKey;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TStockpileStatus;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.util.collection.Nullables;

import static ru.yandex.solomon.labels.protobuf.LabelSelectorConverter.selectorsToNewProto;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
class Converter {
    private final ArchiveConverter archiveConverter;

    Converter(ArchiveConverter archiveConverter) {
        this.archiveConverter = archiveConverter;
    }

    // Need toProto for request and both toProto & fromProto for response

    /**
     * FIND
     */

    public TFindRequest toProto(FindRequest request) {
        return TFindRequest.newBuilder()
                .setSelectors(selectorsToNewProto(request.getSelectors()))
                .setLimit(request.getLimit())
                .setOffset(request.getOffset())
                .setUseNewFormat(request.isUseNewFormat())
                .build();
    }

    public TFindResponse toProto(FindResponse response) {
        return TFindResponse.newBuilder()
                .setStatus(toProto(response.getStatus()))
                .addAllMetrics(response.getMetrics().stream().map(this::toProto).collect(Collectors.toList()))
                .setTotalCount(response.getTotalCount())
                .putAllMetricsByDestination(response.getMetricsCountByDestination())
                .setTruncated(response.isTruncated())
                .setAllDestinationSuccess(response.isAllDestSuccess())
                .build();
    }

    public FindResponse fromProto(TFindResponse response) {
        return new FindResponse(
                fromProto(response.getStatus()),
                fromProto(response.getMetricsList()),
                response.getTotalCount(),
                response.getMetricsByDestinationMap(),
                response.getTruncated(),
                response.getAllDestinationSuccess());
    }

    /**
     * READ
     */

    public TReadRequest toProto(ReadRequest request) {
        return TReadRequest.newBuilder()
                .setMetricKey(toProto(request.getKey()))
                .setFromMillis(request.getFromMillis())
                .setToMillis(request.getToMillis())
                .setGridMillis(request.getGridMillis())
                .setAggregation(request.getAggregation())
                .build();
    }

    public TReadResponse toProto(ReadResponse response) {
        return TReadResponse.newBuilder()
                .setMetricKey(toProto(response.getMetricKey()))
                .setStatus(toProto(response.getStatus()))
                .setSource(toProto(response.getSource()))
                .setAllDestinationSuccess(response.isAllDestSuccess())
                .build();
    }

    public ReadResponse fromProto(TReadResponse response) {
        var metricKey = fromProto(response.getMetricKey());
        var archive = response.getSource();
        var type = archive.getHeader().getType();
        var metric = new Metric<>(metricKey, type, archiveConverter.fromProto(archive));
        return new ReadResponse(metric, fromProto(response.getStatus()), response.getAllDestinationSuccess());
    }

    /**
     * UTIL
     */

    private TMetricKey toProto(MetricKey key) {
        return TMetricKey.newBuilder()
                .setName(key.getName())
                .setType(MetricTypeConverter.toNotNullProto(key.getType()))
                .setLabels(LabelConverter.labelsToProto(key.getLabels()))
                .setCreatedAtMillis(key.getCreatedAtMillis())
                .addAllStockpileKeys(key.getStockpileKeys().stream()
                        .map(this::toProto)
                        .collect(Collectors.toList()))
                .build();
    }

    private MetricKey fromProto(TMetricKey key) {
        return new MetricKey(
                MetricTypeConverter.fromProto(key.getType()),
                key.getName(),
                LabelConverter.protoToLabels(key.getLabels()),
                key.getCreatedAtMillis(),
                key.getStockpileKeysList().stream()
                        .map(this::fromProto)
                        .collect(Collectors.toList()));
    }

    private List<MetricKey> fromProto(List<TMetricKey> list) {
        switch (list.size()) {
            case 0:
                return List.of();
            case 1:
                return List.of(fromProto(list.get(0)));
            default:
                var result = new ArrayList<MetricKey>(list.size());
                for (var proto : list) {
                    result.add(fromProto(proto));
                }
                return result;
        }
    }

    private TStockpileKey toProto(StockpileKey stockpileKey) {
        return TStockpileKey.newBuilder()
                .setDestination(stockpileKey.getDestination())
                .setShardId(stockpileKey.getShardId())
                .setLocalId(stockpileKey.getLocalId())
                .build();
    }

    private StockpileKey fromProto(TStockpileKey stockpileKey) {
        return new StockpileKey(
                stockpileKey.getDestination(),
                stockpileKey.getShardId(),
                stockpileKey.getLocalId());
    }

    private TMetabaseStatus toProto(MetabaseStatus status) {
        return TMetabaseStatus.newBuilder()
                .setCode(status.getCode())
                .setMessage(Nullables.orEmpty(status.getDescription()))
                .build();
    }

    private MetabaseStatus fromProto(TMetabaseStatus status) {
        return new MetabaseStatus(status.getCode(), status.getMessage());
    }

    private TStockpileStatus toProto(StockpileStatus status) {
        return TStockpileStatus.newBuilder()
                .setCode(status.getCode())
                .setMessage(Nullables.orEmpty(status.getDescription()))
                .build();
    }

    private StockpileStatus fromProto(TStockpileStatus status) {
        return new StockpileStatus(status.getCode(), status.getMessage());
    }

    private StockpileCanonicalProto.Archive toProto(AggrGraphDataIterable source) {
        if (source instanceof MetricArchiveGeneric) {
            return archiveConverter.toProto((MetricArchiveGeneric) source);
        }
        return toProto(MetricArchiveImmutable.of(source));
    }

}
