package ru.yandex.solomon.metrics.client;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.combined.CombinedCall;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindCapture;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindRequest;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindResponse;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TMetricsClientCapture;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadCapture;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadRequest;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadResponse;
import ru.yandex.solomon.selfmon.AvailabilityStatus;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class RecordingMetricsClient implements MetricsClient {
    private final MetricsClient proxy;
    private final Converter converter;

    private final ConcurrentMap<TFindRequest, TFindResponse> findCaptures = new ConcurrentHashMap<>();
    private final ConcurrentMap<TReadRequest, TReadResponse> readCaptures = new ConcurrentHashMap<>();

    private UnsupportedOperationException uoe() {
        return new UnsupportedOperationException();
    }

    private RecordingMetricsClient(MetricsClient proxy, ArchiveConverter archiveConverter) {
        this.proxy = proxy;
        this.converter = new Converter(archiveConverter);
    }

    public static RecordingMetricsClient wrap(MetricsClient other, ArchiveConverter archiveConverter) {
        return new RecordingMetricsClient(other, archiveConverter);
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        return proxy.find(request)
                .whenComplete((response, e) -> {
                    if (e == null) {
                        var req = converter.toProto(request);
                        findCaptures.put(req, converter.toProto(response));
                    }
                });
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request) {
        return proxy.uniqueLabels(request);
    }

    @Override
    public CompletableFuture<ReadResponse> read(ReadRequest request) {
        return proxy.read(request)
                .whenComplete((response, e) -> {
                    if (e == null) {
                        var req = converter.toProto(request);
                        readCaptures.put(req, converter.toProto(response));
                    }
                });
    }


    @Override
    public Collection<String> getDestinations() {
        return proxy.getDestinations();
    }

    @Override
    public void close() {
        proxy.close();
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOneWithName(ResolveOneWithNameRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveManyWithName(ResolveManyWithNameRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<LabelNamesResponse> labelNames(LabelNamesRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<LabelValuesResponse> labelValues(LabelValuesRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<FindAndReadManyResponse> findAndReadMany(FindAndReadManyRequest request) {
        return CombinedCall.defaultFindAndReadMany(this, request);
    }

    @Override
    public AvailabilityStatus getMetabaseAvailability() {
        return proxy.getMetabaseAvailability();
    }

    @Override
    public AvailabilityStatus getStockpileAvailability() {
        return proxy.getStockpileAvailability();
    }

    @Override
    public Stream<Labels> metabaseShards(String destination, Selectors selector) {
        return proxy.metabaseShards(destination, selector);
    }

    @Override
    public String getStockpileHostForShardId(String destination, int shardId) {
        throw uoe();
    }

    public TMetricsClientCapture getCapture() {
        return TMetricsClientCapture.newBuilder()
                .addAllDestinations(getDestinations())
                .addAllFindCaptures(findCaptures.entrySet().stream()
                        .map(e -> TFindCapture.newBuilder()
                                    .setRequest(e.getKey())
                                    .setResponse(e.getValue())
                                    .build())
                        .collect(Collectors.toList()))
                .addAllReadCaptures(readCaptures.entrySet().stream()
                        .map(e -> TReadCapture.newBuilder()
                                .setRequest(e.getKey())
                                .setResponse(e.getValue())
                                .build())
                        .collect(Collectors.toList()))
                .build();
    }
}
