package ru.yandex.solomon.metrics.client;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metrics.client.combined.CombinedCall;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindCapture;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindRequest;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TFindResponse;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TMetricsClientCapture;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadCapture;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadRequest;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TReadResponse;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ReplayMetricsClient implements MetricsClient {
    private static final Logger logger = LoggerFactory.getLogger(ReplayMetricsClient.class);

    private final Collection<String> destinations;
    private final Converter converter;

    private final Map<TFindRequest, TFindResponse> findDb;
    private final Map<TReadRequest, TReadResponse> readDb;
    private final boolean useStubDataForStockpile;

    public ReplayMetricsClient(TMetricsClientCapture capture, ArchiveConverter archiveConverter) {
        this(capture, archiveConverter, false);
    }

    public ReplayMetricsClient(TMetricsClientCapture capture, ArchiveConverter archiveConverter, boolean useStubDataForStockpile) {
        this.useStubDataForStockpile = useStubDataForStockpile;
        this.destinations = capture.getDestinationsList();
        this.converter = new Converter(archiveConverter);
        this.findDb = capture.getFindCapturesList()
                .stream()
                .collect(Collectors.toMap(TFindCapture::getRequest, TFindCapture::getResponse));
        this.readDb = capture.getReadCapturesList()
                .stream()
                .collect(Collectors.toMap(TReadCapture::getRequest, TReadCapture::getResponse));
    }

    private UnsupportedOperationException uoe() {
        return new UnsupportedOperationException();
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        return CompletableFuture.completedFuture(syncFind(request));
    }

    private FindResponse syncFind(FindRequest request) {
        var req = converter.toProto(request);
        var resp = findDb.get(req);
        if (resp == null) {
            return new FindResponse(MetabaseStatus.fromCode(
                    EMetabaseStatusCode.INTERNAL_ERROR, "not recorded"));
        }
        return converter.fromProto(resp);
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ReadResponse> read(ReadRequest request) {
        return CompletableFuture.completedFuture(syncRead(request));
    }

    private ReadResponse syncRead(ReadRequest request) {
        var req = converter.toProto(request);
        var resp = readDb.get(req);
        if (resp == null) {
            if (useStubDataForStockpile) {
                logger.error("Using stub data for metric " + request.getKey().getName() + request.getKey().getLabels());
                return new ReadResponse(
                        request.getKey(),
                        StockpileStatus.OK,
                        MetricType.DGAUGE,
                        EProjectId.SOLOMON,
                        makeRandomData(req.getFromMillis(), req.getToMillis()),
                        true);
            } else {
                return new ReadResponse(request.getKey(), StockpileStatus.fromCode(
                        EStockpileStatusCode.INTERNAL_ERROR, "not recorded"));
            }
        }
        return converter.fromProto(resp);
    }

    private static AggrGraphDataArrayList makeRandomData(long fromMillis, long toMillis) {
        var rng = ThreadLocalRandom.current();
        AggrGraphDataArrayList arrayList = new AggrGraphDataArrayList();
        for (long ts = InstantUtils.truncate(fromMillis, 15_000); ts < toMillis; ts += 15_000) {
            arrayList.addRecordShort(ts, rng.nextGaussian() * 1e10);
        }
        return arrayList;
    }

    @Override
    public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request) {
        return CompletableFuture.completedFuture(syncReadMany(request));
    }

    public ReadManyResponse syncReadMany(ReadManyRequest request) {
        List<Operation> ops = request.getOperations();

        Assert.assertEquals(1, ops.size());
        Operation op = ops.get(0);

        Assert.assertTrue(op.hasDownsampling());
        Aggregation aggr = op.getDownsampling().getAggregation();
        long gridMillis = op.getDownsampling().getGridMillis();

        var responses = request.getKeys().stream()
                .sorted(Comparator.comparing(MetricKey::hashCode))
                .map(key -> ReadRequest.newBuilder()
                        .setAggregation(aggr)
                        .setGridMillis(gridMillis)
                        .setFromMillis(request.getFromMillis())
                        .setToMillis(request.getToMillis())
                        .setKey(key)
                        .build())
                .map(this::syncRead)
                .collect(Collectors.toList());

        var failed = responses.stream()
                .filter(resp -> !resp.isOk())
                .findFirst();

        if (failed.isPresent()) {
            return new ReadManyResponse(failed.get().getStatus());
        }

        var metrics = responses.stream()
                .map(ReadResponse::getMetric)
                .collect(Collectors.toList());

        return new ReadManyResponse(metrics);
    }

    @Override
    public CompletableFuture<FindAndReadManyResponse> findAndReadMany(FindAndReadManyRequest request) {
        return CombinedCall.defaultFindAndReadMany(this, request);
    }

    @Override
    public Collection<String> getDestinations() {
        return destinations;
    }

    @Override
    public void close() {
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOneWithName(ResolveOneWithNameRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveManyWithName(ResolveManyWithNameRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<LabelNamesResponse> labelNames(LabelNamesRequest request) {
        throw uoe();
    }

    @Override
    public CompletableFuture<LabelValuesResponse> labelValues(LabelValuesRequest request) {
        throw uoe();
    }

    @Override
    public AvailabilityStatus getMetabaseAvailability() {
        throw uoe();
    }

    @Override
    public Stream<Labels> metabaseShards(String destination, Selectors selector) {
        throw uoe();
    }

    @Override
    public AvailabilityStatus getStockpileAvailability() {
        throw uoe();
    }

    @Override
    public String getStockpileHostForShardId(String destination, int shardId) {
        throw uoe();
    }
}
