package ru.yandex.solomon.metrics.client;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.Nonnull;

import io.netty.util.concurrent.DefaultThreadFactory;

import ru.yandex.metabase.client.impl.MetabaseClientStub;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonClientStub implements AutoCloseable {
    @Nonnull
    private final ExecutorService executorService;
    @Nonnull
    private final StockpileClientStub stockpile;
    @Nonnull
    private final MetabaseClientStub metabase;

    private final boolean verbose;

    public SolomonClientStub() {
        this(true);
    }

    public SolomonClientStub(boolean verbose) {
        this.executorService = Executors.newFixedThreadPool(2,
                new DefaultThreadFactory("response-handle-thread-pool")
        );

        this.stockpile = new StockpileClientStub(executorService);
        this.metabase = new MetabaseClientStub(executorService);

        this.verbose = verbose;
    }

    @Nonnull
    public StockpileClientStub getStockpile() {
        return stockpile;
    }

    @Nonnull
    public MetabaseClientStub getMetabase() {
        return metabase;
    }

    public MetricId metricId(int shardId, long localId) {
        return MetricId.newBuilder().setShardId(shardId).setLocalId(localId).build();
    }

    public MetricId randomMetricId() {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        return metricId(random.nextInt(1, 2000), random.nextLong(1, 15_000_000));
    }

    public void addMetric(Labels labels, AggrGraphDataArrayList source) {
        addMetric(randomMetricId(), "", labels, StockpileColumns.typeByMask(source.columnSetMask()), source);
    }

    public void addMetric(Labels labels, MetricType type, AggrGraphDataArrayList source) {
        addMetric(randomMetricId(), "", labels, type, source);
    }

    public void removeMetrics(Selectors selector) {
        metabase.removeMetrics(selector);
    }

    public void addMetric(MetricId metricId, Labels labels, AggrGraphDataArrayList source) {
        addMetric(metricId, "", labels, StockpileColumns.typeByMask(source.columnSetMask()), source);
    }

    public void addMetric(MetricId metricId, String name, Labels labels, MetricType type, AggrGraphDataArrayList source) {
        metabase.addMetrics(Metric.newBuilder()
                .setMetricId(metricId)
                .setType(type)
                .setName(name)
                .addAllLabels(LabelConverter.labelsToProtoList(labels))
                .build());

        stockpile.addTimeSeries(metricId, source);
        if (verbose) {
            System.out.println("Add timeSeries " + labels + " " + source);
        }
    }

    @Override
    public void close() {
        stockpile.close();
        metabase.close();
        executorService.shutdown();
    }
}
