package ru.yandex.solomon.metrics.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;
import javax.annotation.WillNotClose;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.util.concurrent.DefaultThreadFactory;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.MetricArchiveUtils;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.operation.OperationsPipeline;
import ru.yandex.solomon.math.operation.OperationsPipelineMap;
import ru.yandex.solomon.math.operation.map.OperationDownsampling;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.TimeFilterAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.aggregation.AggregateConverters;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.slog.LogDataIterator;
import ru.yandex.solomon.slog.LogsIndexSerializer;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.slog.ResolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaRecord;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.solomon.util.time.Interval;
import ru.yandex.stockpile.api.DeleteMetricRequest;
import ru.yandex.stockpile.api.DeleteMetricResponse;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricData;
import ru.yandex.stockpile.api.TAllocateLocalIdsRequest;
import ru.yandex.stockpile.api.TAllocateLocalIdsResponse;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TCompressedReadResponse;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TCompressedWriteResponse;
import ru.yandex.stockpile.api.TDeleteMetricByShardRequest;
import ru.yandex.stockpile.api.TDeleteMetricByShardRequest.TOwnerShard;
import ru.yandex.stockpile.api.TDeleteMetricByShardResponse;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.api.TShardCommandRequest;
import ru.yandex.stockpile.api.TShardCommandResponse;
import ru.yandex.stockpile.api.TWriteDataBinaryRequest;
import ru.yandex.stockpile.api.TWriteDataBinaryResponse;
import ru.yandex.stockpile.api.TWriteLogRequest;
import ru.yandex.stockpile.api.TWriteLogResponse;
import ru.yandex.stockpile.client.mem.AccumulatedShardCommand;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequest;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class StockpileClientStub extends AbstractStockpileClientStub {
    private static final Logger logger = LoggerFactory.getLogger(StockpileClientStub.class);

    private final ConcurrentMap<ProducerId, Long> producerSeqNoById;
    private final ConcurrentMap<MetricId, MetricArchiveImmutable> metricIdToTimeSeries;
    private final ConcurrentMap<MetricId, EStockpileStatusCode> metricIdToStockpileStatusCode;
    private final ConcurrentMap<Integer, EStockpileStatusCode> shardIdToStatusCode;
    private volatile EStockpileStatusCode predefineStatusCode = EStockpileStatusCode.OK;
    private volatile AvailabilityStatus availability = AvailabilityStatus.AVAILABLE;
    private final AtomicReference<CountDownLatch> requestSync = new AtomicReference<>(new CountDownLatch(1));

    @WillNotClose
    private final ExecutorService responseHandleExecutorService;
    @WillCloseWhenClosed
    private final ExecutorService executorService;
    private volatile int shardCount = 4096;

    public StockpileClientStub(ExecutorService responseHandleExecutorService) {
        this.responseHandleExecutorService = responseHandleExecutorService;
        this.executorService = Executors.newFixedThreadPool(2,
                new DefaultThreadFactory("server-side-stockpile-thread-pool")
        );
        this.metricIdToTimeSeries = new ConcurrentHashMap<>(20);
        this.metricIdToStockpileStatusCode = new ConcurrentHashMap<>(3);
        this.shardIdToStatusCode = new ConcurrentHashMap<>();
        this.producerSeqNoById = new ConcurrentHashMap<>();
    }

    public void setShardCount(int shardCount) {
        this.shardCount = shardCount;
    }

    public void addTimeSeries(MetricId metricId, AggrGraphDataArrayList timeSeries) {
        MetricArchiveImmutable immutable = MetricArchiveMutable.of(timeSeries).toImmutableNoCopy();
        metricIdToTimeSeries.put(metricId, immutable);
        logger.debug("addTimeSeries {}/{} - {}", metricId.getShardId(), metricId.getLocalId(), immutable);
    }

    public void addTimeSeries(int shardId, long localId, MetricArchiveImmutable archive) {
        var metricId = MetricId.newBuilder()
                .setShardId(shardId)
                .setLocalId(localId)
                .build();

        addTimeSeries(metricId, archive);
    }

    public void addTimeSeries(MetricId id, MetricArchiveImmutable archive) {
        metricIdToTimeSeries.put(id, archive);
    }

    public MetricId randomMetricId() {
        return MetricId.newBuilder()
                .setShardId(randomShardId())
                .setLocalId(StockpileLocalId.random())
                .build();
    }

    public void predefineStatusCodeForMetric(MetricId metricId, EStockpileStatusCode code) {
        metricIdToStockpileStatusCode.put(metricId, code);
    }

    public void predefineStatusCode(EStockpileStatusCode code) {
        this.predefineStatusCode = code;
    }

    public void predefineStatusCode(int shardId, EStockpileStatusCode code) {
        shardIdToStatusCode.put(shardId, code);
    }

    public void setAvailability(AvailabilityStatus availability) {
        this.availability = availability;
    }

    @Nullable
    public MetricArchiveImmutable getTimeSeries(int shardId, long localId) {
        return getTimeSeries(MetricId.newBuilder()
            .setShardId(shardId)
            .setLocalId(localId)
            .build());
    }

    @Nullable
    public MetricArchiveImmutable getTimeSeries(MetricId id) {
        return metricIdToTimeSeries.get(id);
    }

    public int getTimeSeriesCount() {
        return metricIdToTimeSeries.size();
    }

    @Override
    public CompletableFuture<TShardCommandResponse> bulkShardCommand(TShardCommandRequest request) {
        return before().thenApplyAsync(ignore -> request.getCommandsList().stream()
                .map(command -> {
                    switch (command.getRequestCase()) {
                        case DELETE_METRIC:
                            return deleteMetric(command.getDeleteMetric())
                                    .thenApply(response -> commandResponse(response.getStatus(), request.getShardId()));
                        case WRITE:
                            return writeOne(command.getWrite())
                                    .thenApply(response -> commandResponse(response.getStatus(), request.getShardId()));
                        case DELETE_METRIC_DATA:
                            return deleteMetricData(command.getDeleteMetricData())
                                    .thenApply(response -> commandResponse(response.getStatus(), request.getShardId()));
                        case COMPRESSEDWRITE:
                            return writeCompressedOne(command.getCompressedWrite())
                                    .thenApply(response -> commandResponse(response.getStatus(), request.getShardId()));
                        default:
                            throw new UnsupportedOperationException("Not implemented yet: " + command.getRequestCase());
                    }
                })
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf)))
                .thenCompose(future -> future)
                .thenApply(responses -> {
                    for (var resp : responses) {
                        if (resp.getStatus() != EStockpileStatusCode.OK) {
                            return resp;
                        }
                    }

                    return commandResponse(EStockpileStatusCode.OK, request.getShardId());
                });
    }

    private TShardCommandResponse commandResponse(EStockpileStatusCode status, int shardId) {
        return TShardCommandResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setShardId(shardId)
                .build();
    }

    @Override
    public CompletableFuture<TShardCommandResponse> bulkShardCommand(AccumulatedShardCommand commands) {
        return before().thenApplyAsync(ignore -> {
            try {
                ByteBufInputStream stream = new ByteBufInputStream(commands.getReadOnlyBuffer());
                TShardCommandRequest request = TShardCommandRequest.parseFrom(stream);
                return bulkShardCommand(request);
            } catch (IOException e) {
                return CompletableFuture.completedFuture(TShardCommandResponse.newBuilder()
                    .setShardId(commands.getShardId())
                    .setStatus(EStockpileStatusCode.INTERNAL_ERROR)
                    .setStatusMessage(Throwables.getStackTraceAsString(e))
                    .build());
            } finally {
                commands.close();
            }
        }, executorService).thenCompose(f -> f);
    }

    @Override
    public CompletableFuture<TCompressedWriteResponse> writeCompressedOne(TCompressedWriteRequest request) {
        return before().thenApplyAsync(ignore -> {
            var metricId = request.getMetricId();
            EStockpileStatusCode predefinedStatus = statusCode(metricId);
            if (predefinedStatus != EStockpileStatusCode.OK) {
                return TCompressedWriteResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(predefinedStatus)
                    .setStatusMessage("Predefined status code on stub")
                    .build();
            }

            StockpileFormat format = StockpileFormat.byNumber(request.getBinaryVersion());
            for (var chunk : request.getChunksList()) {
                MetricArchiveImmutable archive = MetricArchiveNakedSerializer.serializerForFormatSealed(format)
                    .deserializeToEof(new StockpileDeserializer(chunk.getContent()));

                logger.debug("writeCompressedOne {}/{} - {}", metricId.getShardId(), metricId.getLocalId(), archive);
                metricIdToTimeSeries.compute(metricId, (id, prev) -> {
                    var mutable = prev != null ? prev.toMutable() : new MetricArchiveMutable();
                    mutable.updateWith(archive);
                    mutable.addAll(archive);
                    return mutable.toImmutableNoCopy();
                });
            }
            return TCompressedWriteResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .build();
        }, executorService);
    }

    @Override
    public CompletableFuture<DeleteMetricResponse> deleteMetric(DeleteMetricRequest request) {
        return before().thenApplyAsync(ignore -> {
            final MetricId metricId = request.getMetricId();
            metricIdToTimeSeries.remove(metricId);
            return DeleteMetricResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setMetricId(metricId)
                .build();
        }, executorService);
    }

    @Override
    public CompletableFuture<TCompressedReadResponse> readCompressedOne(TReadRequest request) {
        CompletableFuture<TCompressedReadResponse> future = new CompletableFuture<>();
        before().thenApplyAsync(ignore -> rpcEmulatedReadCompressedOne(request), executorService)
                .whenCompleteAsync((response, throwable) -> {
                    if (throwable != null) {
                        future.complete(TCompressedReadResponse.newBuilder()
                                .setMetricId(request.getMetricId())
                                .setCookie(request.getCookie())
                                .setStatus(EStockpileStatusCode.INTERNAL_ERROR)
                                .setBinaryVersion(request.getBinaryVersion())
                                .setStatusMessage(Throwables.getStackTraceAsString(throwable))
                                .build());
                    } else {
                        future.complete(response);
                    }
                }, responseHandleExecutorService);

        return future;
    }

    @Override
    public CompletableFuture<TCompressedReadManyResponse> readCompressedMany(TReadManyRequest request) {
        return before().thenApplyAsync(ignore -> rpcEmulatedReadCompressedMany(request), executorService)
                .thenComposeAsync(Function.identity(), responseHandleExecutorService)
                .exceptionally(e -> TCompressedReadManyResponse.newBuilder()
                        .setStatus(EStockpileStatusCode.INTERNAL_ERROR)
                        .setStatusMessage(Throwables.getStackTraceAsString(e))
                        .build());
    }

    @Override
    public CompletableFuture<TWriteDataBinaryResponse> writeDataBinary(TWriteDataBinaryRequest request) {
        return before().thenApplyAsync(ignore -> {
            Long2ObjectOpenHashMap<MetricArchiveMutable> dataByLocalId;
            try (var r = StockpileShardWriteRequest.deserialize(request.getContent().toByteArray())) {
                dataByLocalId = r.getDataByLocalId();
            }

            var metricIdBuilder = MetricId.newBuilder()
                .setShardId(request.getShardId());
            for (var e : dataByLocalId.long2ObjectEntrySet()) {
                MetricId metricId = metricIdBuilder.setLocalId(e.getLongKey()).build();
                logger.debug("writeDataBinary {}/{} - {}", metricId.getShardId(), metricId.getLocalId(), e.getValue());
                metricIdToTimeSeries.compute(metricId, (id, prev) -> {
                    var mutable = prev != null ? prev.toMutable() : new MetricArchiveMutable();
                    mutable.updateWith(e.getValue());
                    return mutable.toImmutableNoCopy();
                });
            }

            return TWriteDataBinaryResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .build();
        }, executorService);
    }

    @Override
    public CompletableFuture<TWriteLogResponse> writeLog(TWriteLogRequest request) {
        return before().thenApplyAsync(ignore -> {
            var status = statusCode(request.getShardId());
            if (status != EStockpileStatusCode.OK) {
                return TWriteLogResponse.newBuilder()
                    .setStatus(status)
                    .setStatusMessage("predefined status")
                    .build();
            }

            try {
                int shardId = request.getShardId();
                var index = LogsIndexSerializer.deserialize(ByteStrings.toByteBuf(request.getIndex()));
                var content = ByteStrings.toByteBuf(request.getContent());
                @Nullable
                ParsedLog prevLog = null;
                for (int i = 0; i < index.getSize(); i++) {
                    var meta = content.readSlice(index.getMetaSize(i));
                    var data = content.readSlice(index.getDataSize(i));
                    var log = parseLog(shardId, meta, data);
                    if (prevLog == null) {
                        prevLog = log;
                        continue;
                    }

                    if (prevLog.producerEqual(log)) {
                        prevLog.combine(log);
                    } else {
                        writeLog(prevLog);
                        prevLog = log;
                    }
                }

                if (prevLog != null) {
                    writeLog(prevLog);
                }

                return TWriteLogResponse.newBuilder()
                        .setStatus(EStockpileStatusCode.OK)
                        .build();
            } catch (Throwable e) {
                return TWriteLogResponse.newBuilder()
                        .setStatus(EStockpileStatusCode.INTERNAL_ERROR)
                        .setStatusMessage(Throwables.getStackTraceAsString(e))
                        .build();
            }
        }).whenComplete((ignore, e) -> {
            requestSync.get().countDown();
        });
    }

    private ParsedLog parseLog(int shardId, ByteBuf meta, ByteBuf data) {
        var metricIdBuilder = MetricId.newBuilder().setShardId(shardId);
        var header = new ResolvedLogMetaHeader(meta);
        var producerId = new ProducerId(shardId, header.producerId);

        var result = new ParsedLog(producerId, header.producerSeqNo);
        var metaRecord = new ResolvedLogMetaRecord();
        try(var metaIt = new ResolvedLogMetaIteratorImpl(header, meta);
            var dataIt = LogDataIterator.create(data))
        {
            while (metaIt.next(metaRecord)) {
                MetricId metricId = metricIdBuilder.setLocalId(metaRecord.localId).build();
                var archive = result.archiveById.computeIfAbsent(metricId, ignore -> new MetricArchiveMutable());
                dataIt.next(archive);
            }
        }
        return result;
    }

    private void writeLog(ParsedLog parsed) {
        if (parsed.producerId.producerId != 0) {
            Long prev;
            do {
                prev = producerSeqNoById.putIfAbsent(parsed.producerId, parsed.producerSeqNo);
                if (prev == null) {
                    break;
                }
                if (prev >= parsed.producerSeqNo) {
                    return;
                }
            } while (!producerSeqNoById.replace(parsed.producerId, prev, parsed.producerSeqNo));
        }

        for (var entry : parsed.archiveById.entrySet()) {
            var metricId = entry.getKey();
            var archive = entry.getValue();
            logger.debug("writeLog {}/{} - {}", metricId.getShardId(), metricId.getLocalId(), archive);
            metricIdToTimeSeries.compute(metricId, (id, prev) -> {
                if (prev == null) {
                    return archive.toImmutableNoCopy();
                }
                var mutable = prev.toMutable();
                mutable.updateWith(archive);
                return mutable.toImmutableNoCopy();
            });
        }
    }

    @Override
    public CompletableFuture<TAllocateLocalIdsResponse> allocateLocalIds(TAllocateLocalIdsRequest request) {
        return before().thenApplyAsync(ignore -> {
            var status = statusCode(request.getShardId());
            if (status != EStockpileStatusCode.OK) {
                return TAllocateLocalIdsResponse.newBuilder()
                    .setStatus(status)
                    .setStatusMessage("Predefined status code on stub")
                    .build();
            }

            long[] localIds = new long[request.getSize()];
            for (int index = 0; index < localIds.length; index++) {
                localIds[index] = StockpileLocalId.random();
            }

            return TAllocateLocalIdsResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setList(TAllocateLocalIdsResponse.TList.newBuilder()
                    .addAllLocalIds(LongArrayList.wrap(localIds))
                    .build())
                .build();
        });
    }

    @Override
    public CompletableFuture<TDeleteMetricByShardResponse> deleteMetricByShard(TDeleteMetricByShardRequest request) {
        return before().thenApplyAsync(ignore -> {
            var status = statusCode(request.getShardId());
            if (status != EStockpileStatusCode.OK) {
                return TDeleteMetricByShardResponse.newBuilder()
                        .setStatus(status)
                        .setStatusMessage("Predefined status code on stub")
                        .build();
            }

            Set<TOwnerShard> deletedShards = Set.copyOf(request.getOwnerShardsList());
            var it = metricIdToTimeSeries.values().iterator();
            while (it.hasNext()) {
                var archive = it.next();
                var ownerShard = TOwnerShard.newBuilder()
                        .setProjectId(archive.getOwnerProjectId())
                        .setShardId(archive.getOwnerShardId())
                        .build();

                if (deletedShards.contains(ownerShard)) {
                    it.remove();
                }
            }

            return TDeleteMetricByShardResponse.newBuilder()
                    .setStatus(EStockpileStatusCode.OK)
                    .build();
        });
    }

    @Override
    public int getReadyShardsCount() {
        return getTotalShardsCount();
    }

    @Override
    public int getTotalShardsCount() {
        return shardCount;
    }

    public int randomShardId() {
        return ThreadLocalRandom.current().nextInt(1, shardCount + 1);
    }

    @Override
    public AvailabilityStatus getAvailability() {
        return availability;
    }

    private TCompressedReadResponse rpcEmulatedReadCompressedOne(TReadRequest request) {
        EStockpileStatusCode predefinedStatus = statusCode(request.getMetricId());
        if (predefinedStatus != EStockpileStatusCode.OK) {
            return TCompressedReadResponse.newBuilder()
                    .setMetricId(request.getMetricId())
                    .setStatus(predefinedStatus)
                    .setStatusMessage("Predefined status code on stub")
                    .setBinaryVersion(request.getBinaryVersion())
                    .setCookie(request.getCookie())
                    .build();
        }

        AggrGraphDataArrayList timeSeries = rpcEmulatedRead(request);
        StockpileFormat format = StockpileFormat.byNumber(request.getBinaryVersion());

        return TCompressedReadResponse.newBuilder()
                .setMetricId(request.getMetricId())
                .setStatus(EStockpileStatusCode.OK)
                .setBinaryVersion(format.getFormat())
                .setCookie(request.getCookie())
                .addChunks(TimeSeriesCodecTestSupport.encodeChunk(format, timeSeries))
                .build();
    }

    private CompletableFuture<TCompressedReadManyResponse> rpcEmulatedReadCompressedMany(TReadManyRequest request) {
        List<MetricId> metricIds = new ArrayList<>(request.getLocalIdsCount());
        for (long localId : request.getLocalIdsList()) {
            MetricId metricId = MetricId.newBuilder()
                    .setShardId(request.getShardId())
                    .setLocalId(localId)
                    .build();

            EStockpileStatusCode predefinedStatus = statusCode(metricId);
            if (predefinedStatus != EStockpileStatusCode.OK) {
                return CompletableFuture.completedFuture(TCompressedReadManyResponse.newBuilder()
                        .setStatus(predefinedStatus)
                        .setStatusMessage("Predefined status code on stub")
                        .build());
            }

            metricIds.add(metricId);
        }

        long toMillis = request.getToMillis() != 0
                ? request.getToMillis()
                : System.currentTimeMillis();

        Interval interval = Interval.millis(request.getFromMillis(), toMillis);
        return readMany(metricIds, interval, request.getOperationsList())
                .collect(metric -> toMetricData(metric, request.getBinaryVersion()))
                .thenApply(metrics -> TCompressedReadManyResponse.newBuilder()
                        .setStatus(EStockpileStatusCode.OK)
                        .addAllMetrics(metrics)
                        .build());
    }

    private OperationsPipeline<MetricId> readMany(List<MetricId> ids, Interval interval, List<Operation> operations) {
        return ids.parallelStream()
                .map(metricId -> CompletableFuture.supplyAsync(() -> {
                    MetricArchiveImmutable timeseries =
                            metricIdToTimeSeries.getOrDefault(metricId, MetricArchiveImmutable.empty);

                    AggrGraphDataArrayList filtered = filter(timeseries, interval.getBeginMillis(), interval.getEndMillis());
                    return new Metric<>(metricId, timeseries.getType(), filtered);
                }, executorService))
                .collect(collectingAndThen(toList(), OperationsPipelineMap::new))
                .apply(interval, operations);
    }

    private MetricData toMetricData(Metric<MetricId> metric, int binaryVersion) {
        MetricData.Builder builder = MetricData.newBuilder();

        MetricId metricId = metric.getKey();
        if (metricId != null) {
            builder.setShardId(metricId.getShardId());
            builder.setLocalId(metricId.getLocalId());
        }

        AggrGraphDataIterable timeseries = metric.getTimeseries();
        if (timeseries != null) {
            builder.setCompressed(compress(binaryVersion, metric.getTimeseries()));
        }

        TimeseriesSummary summary = metric.getSummary();
        if (summary != null) {
            AggregateConverters.fillAggregate(builder, metric.getSummary());
        }

        return builder.build();
    }

    private TimeSeries compress(int binaryFormat, AggrGraphDataIterable source) {
        final StockpileFormat format = StockpileFormat.byNumber(binaryFormat);
        MetricArchiveImmutable archive = MetricArchiveUtils.encode(format, source);
        ByteString content = MetricArchiveNakedSerializer.serializerForFormatSealed(format).serializeToByteString(archive);
        return TimeSeries.newBuilder()
                .setFormatVersion(binaryFormat)
                .addChunks(TimeSeries.Chunk.newBuilder()
                        .setPointCount(archive.getRecordCount())
                        .setContent(content)
                        .build())
                .build();
    }

    private AggrGraphDataArrayList rpcEmulatedRead(TReadRequest request) {
        try {
            TimeUnit.MICROSECONDS.sleep(ThreadLocalRandom.current().nextInt(2, 10));
            return read(request);
        } catch (InterruptedException e) {
            throw Throwables.propagate(e);
        }
    }

    private AggrGraphDataArrayList read(TReadRequest request) {
        MetricArchiveImmutable timeseries = metricIdToTimeSeries.getOrDefault(
                request.getMetricId(),
                MetricArchiveImmutable.empty);

        AggrGraphDataArrayList filtered = filter(timeseries, request);
        return downsampling(filtered, request);
    }

    private AggrGraphDataArrayList filter(AggrGraphDataIterable source, TReadRequest request) {
        return filter(source, request.getFromMillis(), request.getToMillis());
    }

    private AggrGraphDataArrayList filter(AggrGraphDataIterable source, long fromMillisInclusive, long toMillisExclusive) {
        if (fromMillisInclusive == 0 && toMillisExclusive == 0) {
            return AggrGraphDataArrayList.of(source);
        }

        var it = source.iterator();
        if (fromMillisInclusive != 0) {
            it = TimeFilterAggrGraphDataIterator.sliceFrom(it, fromMillisInclusive);
        }

        if (toMillisExclusive != 0) {
            it = TimeFilterAggrGraphDataIterator.sliceTo(it, toMillisExclusive - 1);
        }

        return AggrGraphDataArrayList.of(it);
    }

    private AggrGraphDataArrayList downsampling(AggrGraphDataArrayList source, TReadRequest request) {
        if (request.getGridMillis() <= 0 || source.isEmpty()) {
            return source;
        }
        long fromMillis = request.getFromMillis() > 0 ? request.getFromMillis() : source.getTsMillis(0);
        long toMillis = request.getToMillis() > 0 ? request.getToMillis() : source.getTsMillis(source.length() - 1);

        Interval interval = Interval.millis(fromMillis, toMillis);
        var operation = new OperationDownsampling<>(interval, ru.yandex.solomon.math.protobuf.OperationDownsampling.newBuilder()
            .setGridMillis(request.getGridMillis())
            .setAggregation(request.getAggregation())
            .build());
        var result = operation.apply(new Metric<>(null, StockpileColumns.typeByMask(source.columnSetMask()), source));
        return AggrGraphDataArrayList.of(Objects.requireNonNull(result.getTimeseries()));
    }

    public CountDownLatch requestSync() {
        return requestSync.getAndUpdate(prev -> {
            if (prev.getCount() != 0) {
                return prev;
            }
            return new CountDownLatch(1);
        });
    }

    @Override
    public void close() {
        executorService.shutdown();
    }

    private EStockpileStatusCode statusCode(MetricId metric) {
        var status = metricIdToStockpileStatusCode.get(metric);
        if (status == null) {
            status = statusCode(metric.getShardId());
        }

        return Nullables.orDefault(status, EStockpileStatusCode.OK);
    }

    private EStockpileStatusCode statusCode(int shardId) {
        return shardIdToStatusCode.getOrDefault(shardId, predefineStatusCode);
    }

    private static class ProducerId extends DefaultObject {
        private final int shardId;
        private final int producerId;

        public ProducerId(int shardId, int producerId) {
            this.shardId = shardId;
            this.producerId = producerId;
        }
    }

    private static class ParsedLog {
        private final ProducerId producerId;
        private final long producerSeqNo;
        private final Map<MetricId, MetricArchiveMutable> archiveById = new HashMap<>();

        public ParsedLog(ProducerId producerId, long producerSeqNo) {
            this.producerId = producerId;
            this.producerSeqNo = producerSeqNo;
        }


        public boolean producerEqual(ParsedLog log) {
            return producerId.equals(log.producerId) && producerSeqNo == log.producerSeqNo;
        }

        public void combine(ParsedLog log) {
            Preconditions.checkArgument(Objects.equals(producerId, log.producerId));
            for (var entry : log.archiveById.entrySet()) {
                if (!archiveById.containsKey(entry.getKey())) {
                    archiveById.put(entry.getKey(), entry.getValue());
                } else {
                    archiveById.get(entry.getKey()).updateWith(entry.getValue());
                }
            }
        }
    }
}
