package ru.yandex.solomon.metrics.client;

import java.time.Clock;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import org.hamcrest.FeatureMatcher;
import org.hamcrest.Matcher;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationAggregationSummary;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.selfmon.AvailabilityStatus;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.stockpile.api.EStockpileStatusCode;

import static org.hamcrest.CoreMatchers.allOf;
import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;


/**
 * @author Vladimir Gordiychuk
 */
public class CrossDcMetricsClientTest {
    private Unit alice;
    private Unit bob;
    private MetricsClient client;
    private Clock clock;

    @Before
    public void setUp() throws Exception {
        alice = new Unit("alice");
        bob = new Unit("bob");
        clock = new ManualClock();
        client = new CrossDcMetricsClient(clock, ImmutableMap.of(
                alice.name, alice.resolver,
                bob.name, bob.resolver));
    }

    @After
    public void tearDown() throws Exception {
        alice.close();
        bob.close();
    }

    @Test
    public void successResolve() {
        alice.solomon.addMetric(Labels.of("host", "test","sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test","sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor='requestStarted'"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(1));
        assertThat(response.getMetrics().get(0).getLabels(), equalTo(Labels.of("host", "test", "sensor", "requestStarted")));
        assertThat(response.getMetrics().get(0).getStockpileKeys(), iterableWithSize(2));
        assertThat(response.getTotalCount(), equalTo(2));
        assertThat(response.getMetricsCountByDestination().get("alice"), equalTo(1));
        assertThat(response.getMetricsCountByDestination().get("bob"), equalTo(1));
    }

    @Test
    public void combineNotExistsMetrics() {
        Labels h1 = Labels.of("host", "test-01", "sensor", "requestStarted");
        Labels h2 = Labels.of("host", "test-02", "sensor", "requestStarted");

        alice.solomon.addMetric(h1, AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(h2, AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(h1, AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor=='requestStarted'"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(2));

        Map<Labels, MetricKey> keys = response.getMetrics()
                .stream()
                .collect(Collectors.toMap(MetricKey::getLabels, Function.identity()));

        assertThat(keys.get(h1).getStockpileKeys(), iterableWithSize(2));
        assertThat(keys.get(h2).getStockpileKeys(), iterableWithSize(1));

        assertThat(response.getTotalCount(), equalTo(3));
        assertThat(response.getMetricsCountByDestination().get("alice"), equalTo(2));
        assertThat(response.getMetricsCountByDestination().get("bob"), equalTo(1));
    }

    @Test
    public void findLimited() {
        alice.solomon.addMetric(Labels.of("host", "test-1","sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-2", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-3", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-4", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor='requestStarted'"))
                .setLimit(2)
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(2));
        assertThat(response.isTruncated(), equalTo(true));

        assertThat(response.getTotalCount(), equalTo(4));
        assertThat(response.getMetricsCountByDestination().get("alice"), equalTo(3));
        assertThat(response.getMetricsCountByDestination().get("bob"), equalTo(1));
    }

    @Test
    public void findLimitedOnClientSide() {
        alice.solomon.addMetric(Labels.of("host", "test-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-2", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-3", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-4", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor='requestStarted'"))
                .setLimit(3)
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(3));
        assertThat(response.isTruncated(), equalTo(true));

        assertThat(response.getTotalCount(), equalTo(4));
        assertThat(response.getMetricsCountByDestination().get("alice"), equalTo(2));
        assertThat(response.getMetricsCountByDestination().get("bob"), equalTo(2));
    }

    @Test
    public void findLimitedAndShifted() {
        Labels shardLabels = Labels.of("project", "junk", "cluster", "foo", "service", "bar");
        alice.solomon.addMetric(shardLabels.add("host", "test-1"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(shardLabels.add("host", "test-2"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(shardLabels.add("host", "test-3"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(shardLabels.add("host", "test-4"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(shardLabels.add("host", "noise"), AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("project=junk, cluster=foo, service=bar"))
                .setLimit(3)
                .setOffset(2)
                .setDestination("alice")
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(2));
        assertThat(response.getMetrics().get(0).getLabels(), equalTo(shardLabels.add("host", "test-3")));
        assertThat(response.getMetrics().get(1).getLabels(), equalTo(shardLabels.add("host", "test-4")));
        assertThat(response.isTruncated(), equalTo(true));

        assertThat(response.getTotalCount(), equalTo(4));
        assertThat(response.getMetricsCountByDestination().get("alice"), equalTo(4));
    }

    @Test
    public void findShiftedForCrossDc() {
        alice.solomon.addMetric(Labels.of("host", "test-1"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "noise"), AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
            .setSelectors(Selectors.parse("host=*"))
            .setOffset(2)
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.INVALID_REQUEST));
    }

    @Test
    public void findNotTruncated() {
        alice.solomon.addMetric(Labels.of("host", "test-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-4", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor='requestStarted'"))
                .setLimit(3)
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(2));
        assertThat(response.isTruncated(), equalTo(false));
    }

    @Test
    public void findSuccessByAllDest() {
        alice.solomon.addMetric(Labels.of("host", "test-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-4", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        FindResponse response = client.find(FindRequest.newBuilder()
            .setSelectors(Selectors.parse("sensor='requestStarted'"))
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertTrue(response.isAllDestSuccess());
    }

    @Test
    public void findSuccessByOneDest() {
        alice.solomon.addMetric(Labels.of("host", "test-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-4", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);

        FindResponse response = client.find(FindRequest.newBuilder()
            .setSelectors(Selectors.parse("sensor='requestStarted'"))
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertFalse(response.isAllDestSuccess());
    }

    @Test
    public void readAndMergePoints() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5),
                point("2018-04-16T15:00:00Z", 4)
        ));
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        MetricKey key = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5)
        );

        ReadResponse response = client.read(ReadRequest.newBuilder()
                .setKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        assertThat(AggrGraphDataArrayList.of(response.getIterator()), equalTo(expected));
    }

    @Test
    public void readManyMergeTs() {
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5),
                point("2018-04-16T15:00:00Z", 4)
        ));

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        MetricKey keyOne = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));
        MetricKey keyTwo = findOne(Selectors.parse("host='test-02', sensor='requestStarted'"));
        ReadManyResponse response = client.readMany(ReadManyRequest.newBuilder()
                .addKey(keyOne)
                .addKey(keyTwo)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        Map<MetricKey, AggrGraphDataArrayList> result = response.getMetrics()
                .stream()
                .collect(Collectors.toMap(Metric::getKey, o -> AggrGraphDataArrayList.of(o.getTimeseries())));

        assertThat(result.get(keyOne), equalTo(AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5)
        )));

        assertThat(result.get(keyTwo), equalTo(AggrGraphDataArrayList.empty()));
    }

    @Test
    public void readManyMergeSummary() {
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:30Z", 3.5),
                point("2018-04-16T15:00:00Z", 4)
        ));

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        MetricKey keyOne = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));
        MetricKey keyTwo = findOne(Selectors.parse("host='test-02', sensor='requestStarted'"));
        ReadManyResponse response = client.readMany(ReadManyRequest.newBuilder()
                .addKey(keyOne)
                .addKey(keyTwo)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .addOperation(Operation.newBuilder()
                        .setSummary(OperationAggregationSummary.newBuilder()
                                .addAggregations(Aggregation.SUM)
                                .build())
                        .build())
                .addOperation(Operation.newBuilder()
                        .setDropTimeseries(OperationDropTimeSeries.newBuilder().build())
                        .build())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        Map<MetricKey, DoubleSummary> result = response.getMetrics()
                .stream()
                .filter(s -> s.getSummary() != null)
                .collect(Collectors.toMap(Metric::getKey, o -> (DoubleSummary) o.getSummary()));

        assertThat(result.get(keyOne).getSum(), equalTo(2d + 2.5 + 3d));
        assertThat(result.get(keyOne).getCount(), equalTo(3L));

        // I still not ensure that it's valid return null as a result for metric with no data
        assertThat(result.get(keyTwo).getCount(), equalTo(0L));
    }

    @Test
    public void readAtParticularDestination() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5),
                point("2018-04-16T15:00:00Z", 4)
        ));
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        MetricKey key = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5)
        );

        ReadResponse response = client.read(ReadRequest.newBuilder()
                .setKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setDestination(alice.name)
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        assertThat(AggrGraphDataArrayList.of(response.getIterator()), equalTo(expected));
    }

    @Test
    public void readErrorWhenAllReplyWithError() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        alice.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.NODE_UNAVAILABLE);
        bob.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.NODE_UNAVAILABLE);

        MetricKey key = findOne(Selectors.parse("sensor=requestStarted"));

        ReadResponse response = client.read(ReadRequest.newBuilder()
                .setKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.NODE_UNAVAILABLE));
    }

    @Test
    public void readManyErrorWhenAllReplyWithError() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        alice.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.NODE_UNAVAILABLE);
        bob.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.NODE_UNAVAILABLE);

        MetricKey key = findOne(Selectors.parse("sensor=requestStarted"));

        ReadManyResponse response = client.readMany(ReadManyRequest.newBuilder()
                .addKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.NODE_UNAVAILABLE));
    }

    @Test
    public void readSuccessWhenOneOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5),
                point("2018-04-16T15:00:00Z", 4)
        ));

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        bob.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.NODE_UNAVAILABLE);
        MetricKey key = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5)
        );

        ReadResponse response = client.read(ReadRequest.newBuilder()
                .setKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        assertThat(AggrGraphDataArrayList.of(response.getIterator()), equalTo(expected));
    }

    @Test
    public void readManySuccessWhenOneOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5),
                point("2018-04-16T15:00:00Z", 4)
        ));

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 2.5),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        bob.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.NODE_UNAVAILABLE);
        MetricKey key = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T13:31:30Z", 3.5)
        );

        ReadManyResponse response = client.readMany(ReadManyRequest.newBuilder()
                .addKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        AggrGraphDataArrayList result = AggrGraphDataArrayList.of(Iterables.getOnlyElement(response.getMetrics()).getTimeseries());
        assertThat(result, equalTo(expected));
    }

    @Test
    public void findSuccessWhenOneOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        MetricKey key = findOne(Selectors.parse("sensor='requestStarted'"));
        assertThat(key.getLabels(), equalTo(Labels.of("host", "test-01", "sensor", "requestStarted")));
        assertThat(key.getStockpileKeys(), iterableWithSize(1));
    }

    @Test
    public void readFailedWhenBothOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3)
        ));

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:30:30Z", 3)
        ));

        bob.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.DEADLINE_EXCEEDED);
        alice.solomon.getStockpile().predefineStatusCode(EStockpileStatusCode.SHARD_NOT_READY);
        MetricKey key = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));

        ReadResponse response = client.read(ReadRequest.newBuilder()
                .setKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.DEADLINE_EXCEEDED));
        assertThat(response.getStatus().getDescription(), containsString(EStockpileStatusCode.DEADLINE_EXCEEDED.name()));
        assertThat(response.getStatus().getDescription(), containsString(EStockpileStatusCode.SHARD_NOT_READY.name()));
    }

    @Test
    public void findFailedWhenBothOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        alice.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.SHARD_NOT_READY);
        FindResponse response = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-01', sensor='requestStarted'"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.NODE_UNAVAILABLE));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.NODE_UNAVAILABLE.name()));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.SHARD_NOT_READY.name()));
    }

    @Test
    public void findParticularDestination() {
        Labels started = Labels.of("host", "test-01", "sensor", "requestStarted");

        alice.solomon.addMetric(started, AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());

        FindResponse result = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-01'"))
                .setDestination(alice.name)
                .build())
                .join();

        assertThat(result.getStatus(), equalTo(MetabaseStatus.OK));
        assertThat(result.getMetrics(), iterableWithSize(1));
        assertThat(result.getMetrics().get(0).getLabels(), equalTo(started));

        assertThat(result.getTotalCount(), equalTo(1));
        assertThat(result.getMetricsCountByDestination().get("alice"), equalTo(1));
    }

    @Test(expected = CompletionException.class)
    public void findNotExistsDestination() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());

        FindResponse result = client.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-01'"))
                .setDestination("nonExistsDestination")
                .build())
                .join();

        fail("When specified prefer destination that doesn't exists request should fail, its allow find problem as fast as possible, response: " + result);
    }

    @Test
    public void readWhenOnlyOneClusterHaveMetric() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        MetricKey key = findOne(Selectors.parse("host='test-01', sensor='requestStarted'"));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3)
        );

        ReadResponse response = client.read(ReadRequest.newBuilder()
                .setKey(key)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        assertThat(AggrGraphDataArrayList.of(response.getIterator()), equalTo(expected));
    }

    @Test
    public void uniqueLabelsSuccessWhenOneOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        alice.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        UniqueLabelsResponse response = client.uniqueLabels(UniqueLabelsRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor=='requestStarted', host='*'"))
                .setLabels(ImmutableSet.of("host"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getUniqueLabels(), allOf(
                iterableWithSize(1),
                hasItem(Labels.of("host", "test-01"))
        ));
    }

    @Test
    public void uniqueLabelsFailedWhenBothOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        alice.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.SHARD_NOT_READY);
        UniqueLabelsResponse response = client.uniqueLabels(UniqueLabelsRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor=='requestStarted'"))
                .setLabels(ImmutableSet.of("host"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), anyOf(equalTo(EMetabaseStatusCode.NODE_UNAVAILABLE), equalTo(EMetabaseStatusCode.SHARD_NOT_READY)));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.NODE_UNAVAILABLE.name()));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.SHARD_NOT_READY.name()));
    }

    @Test
    public void uniqueLabelsCombineReplyFromDc() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        UniqueLabelsResponse response = client.uniqueLabels(UniqueLabelsRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor=='requestStarted', host='*'"))
                .setLabels(ImmutableSet.of("host"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getUniqueLabels(), allOf(
                hasItem(Labels.of("host", "test-01")),
                hasItem(Labels.of("host", "test-02"))
        ));
    }

    @Test
    public void labelNamesMergeResult() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "name", "mySensor"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("test", "noise"), AggrGraphDataArrayList.empty());

        LabelNamesResponse response = client.labelNames(LabelNamesRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-*'"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getNames(), allOf(
                iterableWithSize(3),
                hasItem("host"),
                hasItem("sensor"),
                hasItem("name")
        ));
    }

    @Test
    public void labelNamesSuccessWhenOneOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "name", "mySensor"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("test", "noise"), AggrGraphDataArrayList.empty());

        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        LabelNamesResponse response = client.labelNames(LabelNamesRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-*'"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getNames(), allOf(
                iterableWithSize(2),
                hasItem("host"),
                hasItem("sensor")
        ));
    }

    @Test
    public void labelNamesFailedWhenBothOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "name", "mySensor"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("test", "noise"), AggrGraphDataArrayList.empty());

        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        alice.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.SHARD_NOT_READY);
        LabelNamesResponse response = client.labelNames(LabelNamesRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-*'"))
                .setDeadline(nextDeadline())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.NODE_UNAVAILABLE));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.NODE_UNAVAILABLE.name()));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.SHARD_NOT_READY.name()));
    }

    @Test
    public void labelValuesMergeResult() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestInFlight"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("test", "noise"), AggrGraphDataArrayList.empty());

        LabelValuesResponse response = client.labelValues(LabelValuesRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-01'"))
                .setName("sensor")
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        LabelStats stats = response.getLabelValuesStats()
                .getStatsByLabelKey()
                .get("sensor");

        assertThat(stats.getValues(), allOf(
                hasItem("requestStarted"),
                hasItem("requestInFlight"),
                hasItem("requestCompleted")
        ));

        Map<String, Integer> metricsCountByDestination = response.getMetricsCountByDestination();
        assertThat(metricsCountByDestination.get("alice"), equalTo(2));
        assertThat(metricsCountByDestination.get("bob"), equalTo(2));
    }

    @Test
    public void labelValuesSuccessWhenOneOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestInFlight"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("test", "noise"), AggrGraphDataArrayList.empty());

        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        LabelValuesResponse response = client.labelValues(LabelValuesRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-01'"))
                .setName("sensor")
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));

        assertThat(response.getLabelValuesStats().getMetricsCount(), equalTo(2));

        LabelStats stats = response.getLabelValuesStats()
                .getStatsByLabelKey()
                .get("sensor");

        assertThat(stats.getValues(), allOf(
                hasItem("requestStarted"),
                hasItem("requestInFlight"),
                not(hasItem("requestCompleted"))
        ));

        Map<String, Integer> metricsCountByDestination = response.getMetricsCountByDestination();
        assertThat(metricsCountByDestination.get("alice"), equalTo(2));
    }

    @Test
    public void labelValuesFailedWhenBothOfClusterNotAvailable() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestInFlight"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test-01", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("test", "noise"), AggrGraphDataArrayList.empty());

        bob.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);
        alice.solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.SHARD_NOT_READY);
        LabelValuesResponse response = client.labelValues(LabelValuesRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-01'"))
                .setName("sensor")
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.NODE_UNAVAILABLE));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.NODE_UNAVAILABLE.name()));
        assertThat(response.getStatus().getDescription(), containsString(EMetabaseStatusCode.SHARD_NOT_READY.name()));

        assertThat(response.getMetricsCountByDestination(), equalTo(Collections.emptyMap()));
    }

    @Test
    public void labelValuesLimitedMergeResult() {
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "elapsedTime", "bin", "10"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "elapsedTime", "bin", "50"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "elapsedTime", "bin", "100"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "elapsedTime", "bin", "inf"), AggrGraphDataArrayList.empty());

        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "elapsedTime", "bin", "10"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "elapsedTime", "bin", "50"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "elapsedTime", "bin", "100"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-02", "sensor", "elapsedTime", "bin", "inf"), AggrGraphDataArrayList.empty());

        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "elapsedTime", "bin", "25"), AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(Labels.of("host", "test-01", "sensor", "elapsedTime", "bin", "65"), AggrGraphDataArrayList.empty());

        bob.solomon.addMetric(Labels.of("test", "noise"), AggrGraphDataArrayList.empty());

        LabelValuesResponse response = client.labelValues(LabelValuesRequest.newBuilder()
                .setSelectors(Selectors.parse("host='test-01'"))
                .setName("bin")
                .setLimit(3)
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));

        assertThat(response.getLabelValuesStats().getMetricsCount(), equalTo(6));

        LabelStats stats = response.getLabelValuesStats()
                .getStatsByLabelKey()
                .get("bin");

        assertThat(stats.getValues(), allOf(
                iterableWithSize(3),
                hasItem("10"),
                hasItem("25"),
                hasItem("50")
        ));

        Map<String, Integer> metricsCountByDestination = response.getMetricsCountByDestination();
        assertThat(metricsCountByDestination.get("alice"), equalTo(6));
    }

    @Test
    public void resolveOneMetricSuccessfully() {
        alice.solomon.addMetric(Labels.of("host", "test", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        ResolveOneResponse response = client.resolveOne(ResolveOneRequest.newBuilder()
            .setLabels(Labels.of("host", "test", "sensor", "requestStarted"))
            .build())
            .join();

        MetricKey metric = response.getMetric();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(metric.getLabels(), equalTo(Labels.of("host", "test", "sensor", "requestStarted")));
        assertThat(metric.getStockpileKeys(), iterableWithSize(2));
        assertThat(metric.getStockpileKeys(), allOf(hasItem(hasDestination("bob")), hasItem(hasDestination("alice"))));
    }

    @Test
    public void resolveOneMetricInDcs() {
        Labels h1 = Labels.of("host", "test-01", "sensor", "requestStarted");
        Labels h2 = Labels.of("host", "test-02", "sensor", "requestStarted");

        alice.solomon.addMetric(h1, AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(h2, AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(h1, AggrGraphDataArrayList.empty());

        ResolveManyResponse response = client.resolveMany(ResolveManyRequest.newBuilder()
            .setLabelsList(Arrays.asList(h1, h2))
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(2));

        Map<Labels, MetricKey> keys = response.getMetrics()
            .stream()
            .collect(Collectors.toMap(MetricKey::getLabels, Function.identity()));

        assertThat(keys.get(h1).getStockpileKeys(), iterableWithSize(2));
        assertThat(keys.get(h2).getStockpileKeys(), iterableWithSize(1));
    }

    @Test
    public void resolveOneNonExistentMetric() {
        alice.solomon.addMetric(Labels.of("sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        ResolveManyResponse response = client.resolveMany(ResolveManyRequest.newBuilder()
            .setLabelsList(Collections.singletonList(Labels.of("sensor", "requestCompleted")))
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(0));
    }

    @Test
    public void resolveMetricsSuccessfully() {
        alice.solomon.addMetric(Labels.of("host", "test", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("host", "test", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        ResolveManyResponse response = client.resolveMany(ResolveManyRequest.newBuilder()
            .setLabelsList(Collections.singletonList(Labels.of("host", "test", "sensor", "requestStarted")))
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(1));
        assertThat(response.getMetrics().get(0).getLabels(), equalTo(Labels.of("host", "test", "sensor", "requestStarted")));
        assertThat(response.getMetrics().get(0).getStockpileKeys(), iterableWithSize(2));
    }

    @Test
    public void resolveDifferentMetricsInDcs() {
        Labels h1 = Labels.of("host", "test-01", "sensor", "requestStarted");
        Labels h2 = Labels.of("host", "test-02", "sensor", "requestStarted");

        alice.solomon.addMetric(h1, AggrGraphDataArrayList.empty());
        alice.solomon.addMetric(h2, AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(h1, AggrGraphDataArrayList.empty());

        ResolveManyResponse response = client.resolveMany(ResolveManyRequest.newBuilder()
            .setLabelsList(Arrays.asList(h1, h2))
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(2));

        Map<Labels, MetricKey> keys = response.getMetrics()
            .stream()
            .collect(Collectors.toMap(MetricKey::getLabels, Function.identity()));

        assertThat(keys.get(h1).getStockpileKeys(), iterableWithSize(2));
        assertThat(keys.get(h2).getStockpileKeys(), iterableWithSize(1));
    }

    @Test
    public void resolveNonExistentMetrics() {
        alice.solomon.addMetric(Labels.of("sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        bob.solomon.addMetric(Labels.of("sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        ResolveManyResponse response = client.resolveMany(ResolveManyRequest.newBuilder()
            .setLabelsList(Collections.singletonList(Labels.of("sensor", "requestCompleted")))
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(0));
    }

    @Test
    public void metabaseReady() {
        AvailabilityStatus status = client.getMetabaseAvailability();
        assertEquals(1.0, status.getAvailability(), 0.0);
    }

    @Test
    public void maxAvailabilityOfMetabase() {
        alice.solomon.getMetabase().setAvailability(new AvailabilityStatus(0.3));
        bob.solomon.getMetabase().setAvailability(new AvailabilityStatus(0.5));

        AvailabilityStatus status = client.getMetabaseAvailability();
        assertEquals(0.5, status.getAvailability(), 0.0);
    }

    @Test
    public void stockpileReady() {
        AvailabilityStatus status = client.getStockpileAvailability();
        assertEquals(1.0, status.getAvailability(), 0.0);
    }

    @Test
    public void maxAvailabilityOfStockpile() {
        alice.solomon.getStockpile().setAvailability(new AvailabilityStatus(0.3));
        bob.solomon.getStockpile().setAvailability(new AvailabilityStatus(0.7));

        AvailabilityStatus status = client.getStockpileAvailability();
        assertEquals(0.7, status.getAvailability(), 0.0);
    }

    private MetricKey findOne(Selectors selectors) {
        return client.find(FindRequest.newBuilder()
                .setSelectors(selectors)
                .build())
                .thenApply(response -> {
                    if (!response.isOk()) {
                        throw new IllegalStateException(response.getStatus().toString());
                    }

                    return Iterables.getOnlyElement(response.getMetrics());
                })
                .join();
    }

    private static Matcher<StockpileKey> hasDestination(String destination) {
        return new FeatureMatcher<StockpileKey, String>(
            equalTo(destination),
            "Matches that stockpile key contains same destination",
            "hasDestination")
        {
            @Override
            protected String featureValueOf(StockpileKey stockpileKey) {
                return stockpileKey.getDestination();
            }
        };
    }

    private AggrPoint point(String time, double value) {
        return AggrPoint.builder()
                .time(time)
                .doubleValue(value)
                .build();
    }

    private long nextDeadline() {
        return clock.millis() + TimeUnit.SECONDS.toMillis(30);
    }

    private static class Unit {
        private final String name;
        private final SolomonClientStub solomon;
        private final MetricsClient resolver;

        Unit(String name) {
            this.name = name;
            this.solomon = new SolomonClientStub();
            this.resolver = new DcMetricsClient(name, solomon.getMetabase(), solomon.getStockpile());
        }

        public void close() {
            solomon.close();
        }
    }
}
