package ru.yandex.solomon.metrics.client;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationCombine;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.labelStats.LabelStats;
import ru.yandex.stockpile.api.EStockpileStatusCode;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasEntry;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.iterableWithSize;
import static ru.yandex.solomon.model.point.AggrPoints.point;

/**
 * @author Vladimir Gordiychuk
 */
public class DcMetricsClientTest {
    @Rule
    public TestName testName = new TestName();
    private SolomonClientStub solomon;
    private DcMetricsClient resolver;

    @Before
    public void setUp() throws Exception {
        solomon = new SolomonClientStub();
        resolver = new DcMetricsClient(
                testName.getMethodName(),
                solomon.getMetabase(),
                solomon.getStockpile());
    }

    @After
    public void tearDown() throws Exception {
        solomon.close();
    }

    @Test
    public void emptyFind() {
        FindResponse response = resolver.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("host=*"))
                .build())
                .join();
        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), emptyIterable());
        assertThat(response.getMetricsCountByDestination(), equalTo(Collections.singletonMap("emptyFind", 0)));
    }

    @Test
    public void find() {
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-2", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        FindResponse response = resolver.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("sensor=='requestStarted', host='*'"))
                .build())
                .join();
        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(2));
        assertThat(response.getMetrics().get(0).getLabels(),
                equalTo(Labels.of("sensor", "requestStarted", "host", "solomon-1")));
        assertThat(response.getMetrics().get(1).getLabels(),
                equalTo(Labels.of("sensor", "requestStarted", "host", "solomon-2")));
        assertThat(response.getMetricsCountByDestination().get("find"), equalTo(2));
    }

    @Test
    public void findUnavailable() {
        solomon.getMetabase().predefineStatusCode(EMetabaseStatusCode.NODE_UNAVAILABLE);

        FindResponse response = resolver.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("host=*"))
                .build())
                .join();
        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.NODE_UNAVAILABLE));
    }

    @Test
    public void read() {
        solomon.addMetric(Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        solomon.addMetric(Labels.of("host", "bob", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 100),
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300),
                point("2018-04-16T15:00:00Z", 400)
        ));

        MetricKey key = resolver.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("host=='alice', sensor=='requestStarted'"))
                .build())
                .thenApply(FindResponse::getMetrics)
                .thenApply(Iterables::getOnlyElement)
                .join();

        assertThat(key.getStockpileKeys(), iterableWithSize(1));

        ReadResponse response = resolver.read(ReadRequest.newBuilder()
                .setDeadline(nextDeadline())
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setKey(key)
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3)
        );

        assertThat(AggrGraphDataArrayList.of(response.getIterator()), equalTo(expected));
    }

    @Test
    public void readManyCrossShardNoOp() {
        solomon.addMetric(Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        solomon.addMetric(Labels.of("host", "bob", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 100),
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300),
                point("2018-04-16T15:00:00Z", 400)
        ));

        MetricKey bobKey = findOneKey("host='bob', sensor=='requestStarted'");
        MetricKey aliceKey = findOneKey("host='alice', sensor=='requestStarted'");

        ReadManyResponse response = resolver.readMany(ReadManyRequest.newBuilder()
                .addKey(aliceKey)
                .addKey(bobKey)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        Map<MetricKey, AggrGraphDataArrayList> result = response.getMetrics()
                .stream()
                .collect(Collectors.toMap(Metric::getKey, o -> AggrGraphDataArrayList.of(o.getTimeseries())));

        AggrGraphDataArrayList expectedAlice = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3)
        );

        AggrGraphDataArrayList expectedBob = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300)
        );

        assertThat(result, hasEntry(aliceKey, expectedAlice));
        assertThat(result, hasEntry(bobKey, expectedBob));
    }

    @Test
    public void readManyShardNoOp() {
        solomon.addMetric(solomon.metricId(42, 100), Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        solomon.addMetric(solomon.metricId(42, 133),Labels.of("host", "bob", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 100),
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300),
                point("2018-04-16T15:00:00Z", 400)
        ));

        MetricKey bobKey = findOneKey("host='bob', sensor=='requestStarted'");
        MetricKey aliceKey = findOneKey("host='alice', sensor=='requestStarted'");

        ReadManyResponse response = resolver.readMany(ReadManyRequest.newBuilder()
                .addKey(aliceKey)
                .addKey(bobKey)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        Map<MetricKey, AggrGraphDataArrayList> result = response.getMetrics()
                .stream()
                .collect(Collectors.toMap(Metric::getKey, o -> AggrGraphDataArrayList.of(o.getTimeseries())));

        AggrGraphDataArrayList expectedAlice = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3)
        );

        AggrGraphDataArrayList expectedBob = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300)
        );

        assertThat(result, hasEntry(aliceKey, expectedAlice));
        assertThat(result, hasEntry(bobKey, expectedBob));
    }

    @Test
    public void readManyCrossShard() {
        solomon.addMetric(Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        solomon.addMetric(Labels.of("host", "bob", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 100),
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300),
                point("2018-04-16T15:00:00Z", 400)
        ));

        MetricKey bobKey = findOneKey("host='bob', sensor=='requestStarted'");
        MetricKey aliceKey = findOneKey("host='alice', sensor=='requestStarted'");

        ReadManyResponse response = resolver.readMany(ReadManyRequest.newBuilder()
                .addKey(aliceKey)
                .addKey(bobKey)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .addOperation(Operation.newBuilder()
                        .setCombine(OperationCombine.newBuilder()
                                .setAggregation(Aggregation.SUM)
                                .build())
                        .build())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(1));

        Metric<MetricKey> metric = Iterables.getOnlyElement(response.getMetrics());
        AggrGraphDataArrayList ts = AggrGraphDataArrayList.of(metric.getTimeseries());

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 202, 2),
                point("2018-04-16T13:31:00Z", 303, 2)
        );

        assertThat(ts, equalTo(expected));
    }

    @Test
    public void readManyCrossShardFail() {
        MetricId aliceMetricId = solomon.metricId(42, 123);
        solomon.addMetric(aliceMetricId, Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        solomon.addMetric(Labels.of("host", "bob", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 100),
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300),
                point("2018-04-16T15:00:00Z", 400)
        ));

        MetricKey bobKey = findOneKey("host='bob', sensor=='requestStarted'");
        MetricKey aliceKey = findOneKey("host='alice', sensor=='requestStarted'");

        solomon.getStockpile().predefineStatusCodeForMetric(aliceMetricId, EStockpileStatusCode.NODE_UNAVAILABLE);

        ReadManyResponse response = resolver.readMany(ReadManyRequest.newBuilder()
                .addKey(aliceKey)
                .addKey(bobKey)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .addOperation(Operation.newBuilder()
                        .setCombine(OperationCombine.newBuilder()
                                .setAggregation(Aggregation.SUM)
                                .build())
                        .build())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.NODE_UNAVAILABLE));
    }

    @Test
    public void readManyCrossShardNoOpFail() {
        MetricId aliceMetricId = solomon.metricId(42, 123);
        solomon.addMetric(aliceMetricId, Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        solomon.addMetric(Labels.of("host", "bob", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 100),
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300),
                point("2018-04-16T15:00:00Z", 400)
        ));

        MetricKey bobKey = findOneKey("host='bob', sensor=='requestStarted'");
        MetricKey aliceKey = findOneKey("host='alice', sensor=='requestStarted'");

        solomon.getStockpile().predefineStatusCodeForMetric(aliceMetricId, EStockpileStatusCode.NODE_UNAVAILABLE);

        ReadManyResponse response = resolver.readMany(ReadManyRequest.newBuilder()
                .addKey(aliceKey)
                .addKey(bobKey)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.NODE_UNAVAILABLE));
    }

    @Test
    public void readManyShard() {
        solomon.addMetric(solomon.metricId(45, 11000), Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 1),
                point("2018-04-16T13:30:00Z", 2),
                point("2018-04-16T13:31:00Z", 3),
                point("2018-04-16T15:00:00Z", 4)
        ));

        solomon.addMetric(solomon.metricId(45, 3321), Labels.of("host", "bob", "sensor", "requestStarted"), AggrGraphDataArrayList.of(
                point("2018-04-16T12:59:00Z", 100),
                point("2018-04-16T13:30:00Z", 200),
                point("2018-04-16T13:31:00Z", 300),
                point("2018-04-16T15:00:00Z", 400)
        ));

        MetricKey bobKey = findOneKey("host='bob', sensor=='requestStarted'");
        MetricKey aliceKey = findOneKey("host='alice', sensor=='requestStarted'");

        ReadManyResponse response = resolver.readMany(ReadManyRequest.newBuilder()
                .addKey(aliceKey)
                .addKey(bobKey)
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .addOperation(Operation.newBuilder()
                        .setCombine(OperationCombine.newBuilder()
                                .setAggregation(Aggregation.SUM)
                                .build())
                        .build())
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.OK));
        assertThat(response.getMetrics(), iterableWithSize(1));

        Metric<MetricKey> metric = Iterables.getOnlyElement(response.getMetrics());
        AggrGraphDataArrayList ts = AggrGraphDataArrayList.of(metric.getTimeseries());

        AggrGraphDataArrayList expected = AggrGraphDataArrayList.of(
                point("2018-04-16T13:30:00Z", 202, 2),
                point("2018-04-16T13:31:00Z", 303, 2));

        assertThat(ts, equalTo(expected));
    }

    private MetricKey findOneKey(String selector) {
        return resolver.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse(selector))
                .build())
                .thenApply(FindResponse::getMetrics)
                .thenApply(Iterables::getOnlyElement)
                .join();
    }

    @Test
    public void readError() {
        MetricId metricId = MetricId.newBuilder()
                .setShardId(32)
                .setLocalId(5444)
                .build();

        solomon.addMetric(metricId, Labels.of("host", "alice", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        MetricKey key = resolver.find(FindRequest.newBuilder()
                .setSelectors(Selectors.parse("host=='alice', sensor=='requestStarted'"))
                .build())
                .thenApply(FindResponse::getMetrics)
                .thenApply(Iterables::getOnlyElement)
                .join();

        assertThat(key.getStockpileKeys(), iterableWithSize(1));
        solomon.getStockpile().predefineStatusCodeForMetric(metricId, EStockpileStatusCode.NODE_UNAVAILABLE);

        ReadResponse response = resolver.read(ReadRequest.newBuilder()
                .setDeadline(nextDeadline())
                .setFrom(Instant.parse("2018-04-16T13:30:00Z"))
                .setTo(Instant.parse("2018-04-16T13:40:00Z"))
                .setKey(key)
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EStockpileStatusCode.NODE_UNAVAILABLE));
    }

    @Test
    public void labelNames() {
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "elapsedTimeMillis"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "disk", "sda1"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "noise"), AggrGraphDataArrayList.empty());

        LabelNamesResponse response = resolver.labelNames(LabelNamesRequest.newBuilder()
                .setSelectors(Selectors.parse("host=='solomon-1'"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getNames(), allOf(
                iterableWithSize(3),
                hasItem("host"),
                hasItem("disk"),
                hasItem("sensor")
        ));
    }

    @Test
    public void labelValues() {
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "elapsedTimeMillis"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "disk", "sda1"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "noise"), AggrGraphDataArrayList.empty());

        LabelValuesResponse response = resolver.labelValues(LabelValuesRequest.newBuilder()
                .setName("sensor")
                .setSelectors(Selectors.parse("host=='solomon-1'"))
                .build())
                .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getLabelValuesStats().getMetricsCount(), equalTo(4));

        LabelStats stats = response.getLabelValuesStats().getStatsByLabelKey().get("sensor");
        assertThat(stats.getCount(), equalTo(3));
        assertThat(stats.getCount(), equalTo(3));
        assertThat(stats.getValues(), allOf(
                hasItem("elapsedTimeMillis"),
                hasItem("requestStarted"),
                hasItem("requestCompleted")
        ));

        assertThat(response.getMetricsCountByDestination(), equalTo(Collections.singletonMap("labelValues", 4)));
    }

    @Test
    public void resolveOne() {
        Labels labels = Labels.of("host", "solomon-1", "sensor", "requestStarted");

        solomon.addMetric(labels, AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "noise"), AggrGraphDataArrayList.empty());

        ResolveOneResponse response = resolver.resolveOne(ResolveOneRequest.newBuilder()
            .setLabels(labels)
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));

        MetricKey metricKey = response.getMetric();

        assertThat(metricKey.getLabels(), equalTo(labels));
    }

    @Test
    public void resolveOneNonExistentMetric() {
        solomon.addMetric(Labels.of("host", "Man", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());

        Labels labels =  Labels.of("host", "cluster", "sensor", "unknownSensor");

        ResolveOneResponse response = resolver.resolveOne(ResolveOneRequest.newBuilder()
            .setLabels(labels)
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.NOT_FOUND));
    }

    @Test
    public void resolveMany() {
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "noise"), AggrGraphDataArrayList.empty());

        List<Labels> labels = Arrays.asList(
            Labels.of("host", "solomon-1", "sensor", "requestStarted"),
            Labels.of("host", "solomon-1", "sensor", "requestCompleted")
        );

        ResolveManyResponse response = resolver.resolveMany(ResolveManyRequest.newBuilder()
            .setLabelsList(labels)
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));

        List<MetricKey> metrics = response.getMetrics();

        assertThat(metrics.size(), equalTo(2));

        assertThat(response.getMetrics().get(0).getLabels(),
            equalTo(Labels.of("sensor", "requestStarted", "host", "solomon-1")));
        assertThat(response.getMetrics().get(1).getLabels(),
            equalTo(Labels.of("sensor", "requestCompleted", "host", "solomon-1")));
    }

    @Test
    public void resolveManyNonExistentMetric() {
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestStarted"), AggrGraphDataArrayList.empty());
        solomon.addMetric(Labels.of("host", "solomon-1", "sensor", "requestCompleted"), AggrGraphDataArrayList.empty());

        List<Labels> labels = Collections.singletonList(
            Labels.of("host", "solomon-1", "sensor", "unknownSensor")
        );

        ResolveManyResponse response = resolver.resolveMany(ResolveManyRequest.newBuilder()
            .setLabelsList(labels)
            .build())
            .join();

        assertThat(response.getStatus().getCode(), equalTo(EMetabaseStatusCode.OK));
        assertThat(response.getMetrics(), emptyIterable());
    }

    private long nextDeadline() {
        return System.currentTimeMillis() + TimeUnit.SECONDS.toMinutes(30);
    }
}
