package ru.yandex.solomon.metrics.client;

import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TCompressedReadResponse;

import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static ru.yandex.solomon.metrics.client.TimeSeriesCodecTestSupport.decode;

/**
 * @author Vladimir Gordiychuk
 */
public class TimeSeriesCodecTest {

    @Test
    public void emptyIterator() throws Exception {
        AggrGraphDataArrayList result = decode(encode(AggrGraphDataArrayList.empty()));
        assertThat(result.isEmpty(), equalTo(true));
    }

    @Test
    public void decodeSingleChunk() throws Exception {
        AggrGraphDataArrayList source = AggrGraphDataArrayList.of(
                point("2017-09-06T15:11:00Z", 5),
                point("2017-09-06T15:11:15Z", 2),
                point("2017-09-06T15:11:30Z", 8)
        );

        TCompressedReadResponse compressed = encode(source);
        AggrGraphDataArrayList result = decode(compressed);

        assertThat(result, equalTo(source));
    }

    @Test
    public void decodeMultiChunk() throws Exception {
        AggrGraphDataArrayList firstChunk = AggrGraphDataArrayList.of(
                point("2017-09-06T15:00:00Z", 1),
                point("2017-09-06T15:05:00Z", 2),
                point("2017-09-06T15:10:00Z", 3)
        );

        AggrGraphDataArrayList secondChunk = AggrGraphDataArrayList.of(
                point("2017-09-06T16:15:00Z", 1),
                point("2017-09-06T16:20:00Z", 2),
                point("2017-09-06T16:25:00Z", 3)
        );

        AggrGraphDataArrayList thirdChunk = AggrGraphDataArrayList.of(
                point("2017-09-06T17:30:00Z", 7),
                point("2017-09-06T17:35:00Z", 7),
                point("2017-09-06T17:40:00Z", 7)
        );

        AggrGraphDataArrayList expected = merge(firstChunk, secondChunk, thirdChunk);
        AggrGraphDataArrayList result = decode(encode(firstChunk, secondChunk, thirdChunk, AggrGraphDataArrayList.empty()));
        assertThat(result, equalTo(expected));
    }

    private AggrGraphDataArrayList merge(AggrGraphDataArrayList... source) {
        AggrGraphDataArrayList result = new AggrGraphDataArrayList();
        for (AggrGraphDataArrayList list : source) {
            result.addAll(list);
        }

        return result;
    }

    private MetricId randomMetricId() {
        return MetricId.newBuilder().setShardId(13).setLocalId(1002).build();
    }

    private TCompressedReadResponse encode(AggrGraphDataArrayList... source) {
        return TCompressedReadResponse.newBuilder()
                .setStatus(EStockpileStatusCode.OK)
                .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
                .setMetricId(randomMetricId())
                .addAllChunks(Stream.of(source)
                        .map(TimeSeriesCodecTestSupport::encodeChunk)
                        .collect(Collectors.toList()))
                .build();
    }

    public static AggrPoint point(String time, double value) {
        return AggrPoint.builder()
                .time(time)
                .doubleValue(value)
                .build();
    }
}
