package ru.yandex.solomon.metrics.parser;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.metrics.parser.json.TreeParserJson;
import ru.yandex.solomon.metrics.parser.spack.TreeParserSpack;


/**
 * Results of running this benchmark on solomon-dev-myt-00.search.yandex.net
 * (sorted by metrics speed):
 *
 * --[ spack-none ]------------------------------------------------
 *     data size: 1377.075 MiB
 *    parse time: 2886 ms
 * metrics speed: 54.121 mln. s/s
 *   bytes speed: 688.537 MiB/s
 *
 * --[ spack-lz4 ]------------------------------------------------
 *     data size: 409.135 MiB
 *    parse time: 3736 ms
 * metrics speed: 36.081 mln. s/s
 *   bytes speed: 136.378 MiB/s
 *
 * --[ spack-zstd ]------------------------------------------------
 *     data size: 208.893 MiB
 *    parse time: 4933 ms
 * metrics speed: 27.061 mln. s/s
 *   bytes speed: 52.223 MiB/s
 *
 * --[ spack-zlib ]-----------------------------------------------
 *     data size: 236.379 MiB
 *    parse time: 7209 ms
 * metrics speed: 15.463 mln. s/s
 *   bytes speed: 33.768 MiB/s
 *
 * --[ json ]-----------------------------------------------------
 *     data size: 4382.611 MiB
 *    parse time: 47708 ms
 * metrics speed: 2.303 mln. s/s
 *   bytes speed: 93.247 MiB/s
 *
 * @author Sergey Polovko
 */
public class TreeParserBenchmark {

    private static List<ByteBuf> data = new ArrayList<>(6000);
    private static long dataSize = 0;


    public static void main(String[] args) throws IOException {
        if (args.length != 2) {
            System.err.println("Usage: java TreeParserBenchmark {json|spack} <data-dir>");
            System.exit(1);
        }

        TreeParser parser = treeParser(args[0]);

        {
            System.out.print("read data...");
            long start = System.nanoTime();
            readData(args[1]);
            long end = System.nanoTime();
            System.out.printf(" [done] %d ms%n", TimeUnit.NANOSECONDS.toMillis(end - start));
        }

        for (int i = 0; i < 10; i++) {
            System.out.printf("--[ run %d ]------------------------------------------------%n", i);

            MetricCounter c = new MetricCounter();
            System.out.print("parse data...");
            long start = System.nanoTime();
            for (ByteBuf buf : data) {
                buf.markReaderIndex();
                parser.parseAndProcess(Labels.empty(), buf, c, TreeParser.ErrorListenerIgnore.I, TreeParser.FormatListenerIgnore.I, false);
                buf.resetReaderIndex();
            }

            long end = System.nanoTime();
            System.out.println(" [done]");
            System.out.printf("    data size: %.3f MiB%n", (double) dataSize / (1 << 20));
            System.out.printf("   parse time: %d ms%n", + TimeUnit.NANOSECONDS.toMillis(end - start));
            System.out.printf("metrics speed: %.3f mln. s/s%n", (double) c.count / 1e6 / TimeUnit.NANOSECONDS.toSeconds(end - start));
            System.out.printf("  bytes speed: %.3f MiB/s%n", (double) dataSize / (1 << 20) / TimeUnit.NANOSECONDS.toSeconds(end - start));
        }
    }

    private static TreeParser treeParser(String format) {
        switch (format) {
            case "json": return TreeParserJson.I;
            case "spack": return TreeParserSpack.instance();
            default:
                System.err.println("unknown format: " + format);
                System.exit(1);
                return null;
        }
    }

    private static void readData(String dirPath) throws IOException {
        Path path = Paths.get(dirPath);
        try (DirectoryStream<Path> dir = Files.newDirectoryStream(path)) {
            for (Path child : dir) {
                byte[] buf = Files.readAllBytes(child);
                if (buf.length != 0) {
                    data.add(Unpooled.wrappedBuffer(buf));
                    dataSize += buf.length;
                }
            }
        }
    }

    @ParametersAreNonnullByDefault
    private static final class MetricCounter implements MetricConsumer {
        long count = 0;

        @Override
        public void onMetricBegin(MetricType type, Labels labels, boolean memOnly) {
            count++;
        }

        @Override
        public void onPoint(long tsMillis, double value) {
        }

        @Override
        public void onPoint(long tsMillis, long value) {
        }

        @Override
        public void onTimeSeries(TimeSeries timeSeries) {
        }
    }
}
