package ru.yandex.solomon.metrics.parser.json;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.buffer.ByteBuf;

import ru.yandex.solomon.labels.shard.ShardKey;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public enum JacksonJsonParser implements JsonParser {
    I;

    @Override
    public void parse(ByteBuf bytes, MetricJsonConsumer consumer) {
        try {
            new JacksonJsonParserImpl(bytes).read(consumer);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public ShardKey parseShardKeyOrThrow(ByteBuf bytes) {
        ShardKeyConsumer consumer = new ShardKeyConsumer();
        parse(bytes, consumer);
        if (consumer.shardKey != null) {
            return consumer.shardKey;
        }
        throw new IllegalStateException("commonLabels are not specified");
    }

    @Override
    public MetricCommonData getCommonData(ByteBuf bytes) {
        CommonsConsumer commons = new CommonsConsumer();
        parse(bytes, commons);
        return new MetricCommonData(commons.labels, commons.ts);
    }

    @Override
    public void forEachMetric(ByteBuf bytes, Consumer<MetricJson> consumer) {
        parse(bytes, new ForEachConsumer(consumer));
    }

    /**
     * consumes only common data from json stream
     */
    private static final class CommonsConsumer implements MetricJsonConsumer {
        public long ts = 0;
        public Map<String, String> labels = Collections.emptyMap();

        @Override
        public boolean onCommonTs(long ts) {
            this.ts = ts;
            return needContinue();
        }

        @Override
        public boolean onCommonLabels(Map<String, String> labels) {
            this.labels = labels;
            return needContinue();
        }

        private boolean needContinue() {
            return ts == 0 || labels == Collections.<String, String>emptyMap();
        }

        @Override
        public boolean skipMetrics() {
            return true;
        }
    }

    /**
     * consumes only common labels from json stream and convert them into ShardKey
     */
    private static final class ShardKeyConsumer implements MetricJsonConsumer {
        public ShardKey shardKey;

        @Override
        public boolean onCommonTs(long ts) {
            return true;
        }

        @Override
        public boolean onCommonLabels(Map<String, String> labels) {
            shardKey = ShardKey.get(labels);
            return false;
        }

        @Override
        public boolean skipMetrics() {
            return true;
        }
    }

    /**
     * calls delegate consumer on each metric object
     */
    private static final class ForEachConsumer implements MetricJsonConsumer {
        private final Consumer<MetricJson> delegate;

        private ForEachConsumer(Consumer<MetricJson> delegate) {
            this.delegate = delegate;
        }

        @Override
        public boolean onCommonTs(long ts) {
            return true;
        }

        @Override
        public boolean onCommonLabels(Map<String, String> labels) {
            return true;
        }

        @Override
        public boolean skipMetrics() {
            return false;
        }

        @Override
        public void onMetric(MetricJson metric) {
            this.delegate.accept(metric);
        }
    }
}
