package ru.yandex.solomon.metrics.parser.json;

import java.io.IOException;
import java.util.HashMap;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonToken;
import io.netty.buffer.ByteBuf;
import it.unimi.dsi.fastutil.doubles.DoubleArrayList;
import it.unimi.dsi.fastutil.longs.LongArrayList;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.histogram.ExplicitHistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.util.lang.SmallDoubleToString;
import ru.yandex.solomon.util.lang.SmallIntToString;


/**
 * @author Stepan Koltsov
 * @author Maxim Leonov
 */
class JacksonJsonParserImpl {

    private final com.fasterxml.jackson.core.JsonParser jsonReader;

    private JacksonJsonParserImpl(com.fasterxml.jackson.core.JsonParser jsonReader) {
        this.jsonReader = jsonReader;
    }

    public JacksonJsonParserImpl(ByteBuf bytes) {
        this(JacksonUtils.newJsonParser(bytes));
    }

    private JsonParseException locException(String message) {
        return new JsonParseException(null, message + " at " + jsonReader.getCurrentLocation());
    }

    private void nextToken(JsonToken expected) throws IOException {
        JsonToken jsonToken = jsonReader.nextToken();
        if (jsonToken != expected) {
            throw locException("expected: " + expected + ", got: " + jsonToken);
        }
    }

    private void checkToken(JsonToken expected) throws IOException {
        if (!jsonReader.hasToken(expected)) {
            throw locException("expected: " + expected + ", got: " + jsonReader.getCurrentToken());
        }
    }

    @Nonnull
    private String nextString() throws IOException {
        JsonToken jsonToken = jsonReader.nextToken();
        switch (jsonToken) {
            case VALUE_STRING:
                return jsonReader.getText();
            case VALUE_NUMBER_INT:
                return SmallIntToString.toString(jsonReader.getIntValue());
            case VALUE_NUMBER_FLOAT:
                return SmallDoubleToString.toString(jsonReader.getDoubleValue());
            case VALUE_TRUE:
                return "true";
            case VALUE_FALSE:
                return "false";
            default:
                throw locException("unexpected token: " + jsonToken);
        }
    }

    @Nonnull
    private String readMetricName() throws IOException {
        JsonToken jsonToken = jsonReader.nextToken();
        if (jsonToken == JsonToken.VALUE_STRING) {
            String text = jsonReader.getText();
            if (StringUtils.isBlank(text)) {
                throw locException("metric name cannot be blank");
            }

            return text;
        }

        throw locException("unexpected token for metric name: " + jsonToken);
    }

    private HashMap<String, String> readStringStringMap() throws IOException {
        HashMap<String, String> r = new HashMap<>();

        nextToken(JsonToken.START_OBJECT);
        while (jsonReader.nextToken() != JsonToken.END_OBJECT) {
            String name = jsonReader.getCurrentName();
            // https://st.yandex-team.ru/SOLOMON-2098
            String value = nextString();
            r.put(name, value);
        }
        checkToken(JsonToken.END_OBJECT);

        return r;
    }

    private long readTs() throws IOException {
        return JacksonUtils.readTs(jsonReader);
    }

    private long readLong(JsonToken currentToken) throws IOException {
        switch (currentToken) {
            case VALUE_NUMBER_INT:
                return jsonReader.getValueAsLong();
            case VALUE_NUMBER_FLOAT:
                return Math.round(jsonReader.getValueAsDouble());
            default:
                throw locException("expected number, got: " + currentToken);
        }
    }

    private double readDouble(JsonToken currentToken) throws IOException {
        switch (currentToken) {
            case VALUE_NUMBER_INT:
            case VALUE_NUMBER_FLOAT:
                return jsonReader.getValueAsDouble();
            default:
                throw locException("expected number, got: " + currentToken);
        }
    }

    private double readValue() throws IOException {
        JsonToken next = jsonReader.nextToken();
        double value;
        if (next == JsonToken.VALUE_NUMBER_FLOAT || next == JsonToken.VALUE_NUMBER_INT) {
            value = jsonReader.getValueAsDouble();
        } else if (next == JsonToken.VALUE_STRING) {
            try {
                value = Double.parseDouble(jsonReader.getText());
            } catch (NumberFormatException e) {
                throw locException("expecting number for value, got: " + next);
            }
        } else {
            throw locException("expecting number for value, got: " + next);
        }
        return value;
    }

    private HistogramSnapshot readHistogram() throws IOException {
        nextToken(JsonToken.START_OBJECT);

        DoubleArrayList bounds = null;
        LongArrayList buckets = null;
        long infValue = -1;

        while (jsonReader.nextToken() != JsonToken.END_OBJECT) {
            String name = jsonReader.getCurrentName();
            switch (name) {
                case "bounds":
                    bounds = readHistogramBounds();
                    break;
                case "buckets":
                    buckets = readHistogramBuckets();
                    break;
                case "inf":
                    infValue = readLong(jsonReader.nextToken());
                    break;
                default:
                    throw locException("unexpected property name \'" + name + "\' in histogram object");
            }
        }

        if (bounds == null || buckets == null) {
            throw locException("unexpected empty histogram object");
        }

        if (infValue >= 0) {
            bounds.add(Histograms.INF_BOUND);
            buckets.add(infValue);
        }

        try {
            return new ExplicitHistogramSnapshot(bounds.toDoubleArray(), buckets.toLongArray());
        } catch (IllegalArgumentException e) {
            throw locException(e.getMessage());
        }
    }

    private DoubleArrayList readHistogramBounds() throws IOException {
        nextToken(JsonToken.START_ARRAY);

        DoubleArrayList array = new DoubleArrayList(10);
        double prev = -Double.MAX_VALUE;
        JsonToken token;

        while ((token = jsonReader.nextToken()) != JsonToken.END_ARRAY) {
            double bound = readDouble(token);
            if (bound <= prev) {
                throw locException("histogram bounds must be sorted");
            }
            array.add(bound);
            prev = bound;
        }
        checkToken(JsonToken.END_ARRAY);
        return array;
    }

    private LongArrayList readHistogramBuckets() throws IOException {
        nextToken(JsonToken.START_ARRAY);

        LongArrayList array = new LongArrayList(10);
        JsonToken token;

        while ((token = jsonReader.nextToken()) != JsonToken.END_ARRAY) {
            long bucket = readLong(token);
            if (bucket < 0) {
                throw locException("histogram bucket cannot be negative");
            }
            array.add(bucket);
        }
        checkToken(JsonToken.END_ARRAY);
        return array;
    }

    private TimeSeries readTimeSeries() throws IOException {
        TimeSeries r = TimeSeries.empty();
        long prevTsMillis = 0;

        nextToken(JsonToken.START_ARRAY);
        while (jsonReader.nextToken() != JsonToken.END_ARRAY) {
            checkToken(JsonToken.START_OBJECT);

            long tsMillis = 0;
            double value = Double.NaN;
            HistogramSnapshot hist = ExplicitHistogramSnapshot.EMPTY;

            while (jsonReader.nextToken() != JsonToken.END_OBJECT) {
                String name = jsonReader.getCurrentName();
                switch (name) {
                    case "ts":
                        tsMillis = readTs();
                        break;
                    case "value":
                        value = readValue();
                        break;
                    case "hist":
                        hist = readHistogram();
                        break;
                    default:
                        throw locException("unknown timeSeries attr: " + name);
                }
            }

            if (tsMillis == 0) {
                throw locException("ts is not specified");
            }

            if (prevTsMillis > tsMillis) {
                throw locException("timeseries with out of order ts");
            }

            if (hist == ExplicitHistogramSnapshot.EMPTY) {
                if (Double.isNaN(value)) {
                    throw locException("value is not specified");
                }

                r = r.addDouble(tsMillis, value);
            } else {
                r = r.addHistogram(tsMillis, hist);
            }

            prevTsMillis = tsMillis;
            checkToken(JsonToken.END_OBJECT);
        }
        checkToken(JsonToken.END_ARRAY);

        return r;
    }

    private MetricJson readMetricFromOpenObject() throws IOException {
        MetricJson r = new MetricJson();

        checkToken(JsonToken.START_OBJECT);
        while (jsonReader.nextToken() != JsonToken.END_OBJECT) {
            String name = jsonReader.getCurrentName();
            switch (name) {
                case "name":
                    r.setName(readMetricName());
                    break;
                case "labels":
                    r.setLabels(readStringStringMap());
                    break;
                case "ts":
                    r.setTs(readTs());
                    break;
                case "memOnly":
                    r.setMemOnly(jsonReader.nextBooleanValue());
                    break;
                case "value":
                    r.setValue(readValue());
                    break;
                case "hist":
                    r.setHistogram(readHistogram());
                    break;
                case "mode":
                    r.setDeriv("deriv".equals(jsonReader.nextTextValue()));
                    break;
                case "kind":
                case "type":
                    r.setType(readType());
                    break;
                case "timeseries":
                    r.setTimeSeries(readTimeSeries());
                    break;
                default:
                    throw locException("unknown metric attr: " + name);
            }
        }
        checkToken(JsonToken.END_OBJECT);

        return r;
    }

    private MetricType readType() throws IOException {
        String str = jsonReader.nextTextValue();
        if (StringUtils.isEmpty(str)) {
            return MetricType.DGAUGE;
        }
        switch (str.toUpperCase()) {
            case "GAUGE":
            case "DGAUGE":
                return MetricType.DGAUGE;
            case "IGAUGE": return MetricType.IGAUGE;
            case "COUNTER": return MetricType.COUNTER;
            case "RATE": return MetricType.RATE;
            case "HIST": return MetricType.HIST;
            case "HIST_RATE": return MetricType.HIST_RATE;
            default:
                throw locException("unknown metric type: '" + str + '\'');
        }
    }

    private void consumeMetrics(MetricJsonConsumer consumer) throws IOException {
        nextToken(JsonToken.START_ARRAY);
        while (jsonReader.nextToken() != JsonToken.END_ARRAY) {
            consumer.onMetric(readMetricFromOpenObject());
        }
        checkToken(JsonToken.END_ARRAY);
    }

    void read(MetricJsonConsumer consumer) throws IOException {
        nextToken(JsonToken.START_OBJECT);
        while (jsonReader.nextToken() != JsonToken.END_OBJECT) {
            String name = jsonReader.getCurrentName();
            switch (name) {
                case "ts":
                    if (!consumer.onCommonTs(readTs())) {
                        return;
                    }
                    break;
                case "labels":
                case "commonLabels":
                    if (!consumer.onCommonLabels(readStringStringMap())) {
                        return;
                    }
                    break;
                case "sensors":
                case "metrics":
                    if (consumer.skipMetrics()) {
                        jsonReader.skipChildren();
                    } else {
                        consumeMetrics(consumer);
                    }
                    break;
                default:
                    throw locException("invalid attr: " + name);
            }
        }
        checkToken(JsonToken.END_OBJECT);

        // TODO: check EOF
    }
}
