package ru.yandex.solomon.metrics.parser.json;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.buffer.ByteBuf;
import it.unimi.dsi.fastutil.objects.Object2LongMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.monlib.metrics.labels.validate.TooManyLabelsException;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.metrics.parser.HistogramUtils;
import ru.yandex.solomon.metrics.parser.MetricConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.util.InvalidMetricNameException;
import ru.yandex.solomon.util.MetricNameHelper;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class TreeParserJson implements TreeParser {

    private static final Logger logger = LoggerFactory.getLogger(TreeParserJson.class);

    public static final TreeParserJson I = new TreeParserJson(Labels.allocator, "");

    private final LabelAllocator labelAllocator;
    private final String metricNameLabel;

    public TreeParserJson(LabelAllocator labelAllocator, String metricNameLabel) {
        this.labelAllocator = labelAllocator;
        this.metricNameLabel = metricNameLabel;
    }

    @Override
    public void parseAndProcess(Labels commonLabels, ByteBuf bytes,
                                MetricConsumer metricConsumer, ErrorListener errorListener,
                                FormatListener formatListener, boolean onlyNewFormatWrites)
    {
        JsonParser jsonParser = JsonParser.DEFAULT;

        bytes.markReaderIndex();
        MetricCommonData commonData = jsonParser.getCommonData(bytes);
        bytes.resetReaderIndex();

        var context = new ParserContext(commonData.globalTs, metricConsumer, errorListener, formatListener, onlyNewFormatWrites);
        context.labelsBuilder.addAll(commonLabels);
        addAllLabels(context, "", commonData.commonLabels, true);

        final Labels commonLabelsFinal = context.labelsBuilder.build();

        jsonParser.forEachMetric(bytes,
                metric -> processMetric(context, commonLabelsFinal, metric)
        );
    }

    /**
     * @return count of added labels or {@code -1} in case of any errors
     */
    private int addAllLabels(
            ParserContext context,
            String metricName,
            Map<String, String> labels,
            boolean commonLabels)
    {
        if (context.onlyNewFormatWrites && !commonLabels && metricName.isEmpty()) {
            context.errorListener.invalidMetric(InvalidMetricReason.INVALID_SENSOR_NAME);
            return -1;
        }

        int addedCount = 0;
        for (var e : labels.entrySet()) {
            var label = context.labelCache.get(e);
            if (label != null) {
                if (!addLabel(context, label)) {
                    return -1;
                }

                addedCount++;
                continue;
            }

            final String name = e.getKey();
            final String value = e.getValue();

            if (LabelKeys.PROJECT.equals(name) ||
                    LabelKeys.CLUSTER.equals(name) ||
                    LabelKeys.SERVICE.equals(name))
            {
                if (commonLabels) {
                    // skip labels of shard key if any
                    continue;
                } else {
                    context.errorListener.invalidMetric(InvalidMetricReason.INVALID_LABEL_NAME);
                    return -1;
                }
            }

            if (StringUtils.isEmpty(value)) {
                if (LabelKeys.HOST.equals(name)) {
                    // host="" - is a command to remove host label
                    // see https://st.yandex-team.ru/SOLOMON-136
                    context.labelsBuilder.remove(name);
                    continue;
                }

                // treat empty value as no label at all
                continue;
            }

            if (!addLabel(context, name, value)) {
                return -1;
            }
            addedCount++;
        }


        if (!commonLabels) {
            final boolean canAdd;

            try {
                canAdd = MetricNameHelper.canAddMetricNameLabel(context.labelsBuilder, metricNameLabel, metricName);
            } catch (InvalidMetricNameException e) {
                context.errorListener.invalidMetric(InvalidMetricReason.INVALID_SENSOR_NAME);
                return -1;
            }

            if (canAdd) {
                if (!addLabel(context, metricNameLabel, metricName)) {
                    return -1;
                }
                addedCount++;
            }
        }

        return addedCount;
    }

    private boolean addLabel(
            ParserContext context,
            String name,
            String value)
    {
        if (!LabelValidator.isValidName(name)) {
            context.errorListener.invalidMetric(InvalidMetricReason.INVALID_LABEL_NAME);
            return false;
        }

        if (!LabelValidator.isValidValue(value)) {
            context.errorListener.invalidMetric(InvalidMetricReason.INVALID_LABEL_VALUE);
            return false;
        }

        var label = labelAllocator.alloc(name, value);
        context.labelCache.put(Map.entry(name, value), label);
        return addLabel(context, label);
    }

    private boolean addLabel(ParserContext context, Label label) {
        try {
            context.labelsBuilder.add(label);
            return true;
        } catch (TooManyLabelsException ex) {
            System.out.println("too many");
            context.errorListener.invalidMetric(InvalidMetricReason.TOO_MANY_LABELS);
            return false;
        }
    }

    private void processMetric(
            ParserContext context,
            Labels commonLabels,
            MetricJson metric)
    {
        context.labelsBuilder.clear();
        context.labelsBuilder.addAll(commonLabels);

        boolean isNewFormat = !metric.getName().isEmpty();

        context.formatListener.register(isNewFormat);

        int count = addAllLabels(context, metric.getName(), metric.getLabels(), false);
        if (count == -1) {
            return;
        } else if (count == 0) {
            context.errorListener.invalidMetric(InvalidMetricReason.EMPTY_LABELS);
            return;
        }

        // check that lately we can add 3 more labels for project, cluster, service
        if (!LabelsValidator.isCountValid(context.labelsBuilder.size() + 3)) {
            context.errorListener.invalidMetric(InvalidMetricReason.TOO_MANY_LABELS);
            return;
        }

        MetricType type = metric.getType();
        if (type == MetricType.UNKNOWN) {
            type = metric.isDeriv() ? MetricType.RATE : MetricType.DGAUGE;
        }

        boolean memOnly = metric.getMemOnly();

        long instantMillis = metric.getTs();
        if (instantMillis == 0) {
            instantMillis = context.globalTs;
        }

        TimeSeries timeSeries = metric.getTimeSeries();

        if (type == MetricType.HIST || type == MetricType.HIST_RATE) {
            if (timeSeries != null) {
                MetricType bucketType = HistogramUtils.histogramBucketType(type);
                Map<Label, TimeSeries> bucketTimeSeries = HistogramUtils.splitBuckets(timeSeries, labelAllocator);
                context.metricConsumer.ensureCapacity(bucketTimeSeries.size());
                for (Map.Entry<Label, TimeSeries> entry : bucketTimeSeries.entrySet()) {
                    context.metricConsumer.onMetricBegin(bucketType, context.labelsBuilder.add(entry.getKey()).build(), memOnly);
                    context.metricConsumer.onTimeSeries(entry.getValue());
                }
            } else {
                MetricType bucketType = HistogramUtils.histogramBucketType(type);
                Object2LongMap<Label> valuePerBucket = HistogramUtils.splitBuckets(metric.getHistogram(), labelAllocator);
                context.metricConsumer.ensureCapacity(valuePerBucket.size());
                for (Object2LongMap.Entry<Label> entry : valuePerBucket.object2LongEntrySet()) {
                    Label bin = entry.getKey();
                    long value = entry.getLongValue();
                    context.metricConsumer.onMetricBegin(bucketType, context.labelsBuilder.add(bin).build(), memOnly);
                    context.metricConsumer.onPoint(instantMillis, value);
                }
            }
        } else {
            context.metricConsumer.onMetricBegin(type, context.labelsBuilder.build(), memOnly);
            if (timeSeries != null) {
                context.metricConsumer.onTimeSeries(timeSeries);
            } else {
                context.metricConsumer.onPoint(instantMillis, metric.getValue());
            }
        }
    }

    private static class ParserContext {
        private final long globalTs;
        private final MetricConsumer metricConsumer;
        private final ErrorListener errorListener;
        private final FormatListener formatListener;
        private final boolean onlyNewFormatWrites;
        private final LabelsBuilder labelsBuilder = new LabelsBuilder(Labels.MAX_LABELS_COUNT);
        private final Map<Entry<String, String>, Label> labelCache = new HashMap<>();

        public ParserContext(long globalTs, MetricConsumer metricConsumer, ErrorListener errorListener, FormatListener formatListener, boolean onlyNewFormatWrites) {
            this.globalTs = globalTs;
            this.metricConsumer = metricConsumer;
            this.errorListener = errorListener;
            this.formatListener = formatListener;
            this.onlyNewFormatWrites = onlyNewFormatWrites;
        }
    }
}
