package ru.yandex.solomon.metrics.parser.monitoringJson;

import java.io.IOException;
import java.util.HashMap;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonToken;
import io.netty.buffer.ByteBuf;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.metrics.parser.json.JacksonUtils;
import ru.yandex.solomon.metrics.parser.json.MetricJson;
import ru.yandex.solomon.metrics.parser.json.MetricJsonConsumer;
import ru.yandex.solomon.util.time.InstantUtils;


/**
 * @author Oleg Baryshnikov
 */
class MonitoringJacksonJsonParserImpl {

    private final com.fasterxml.jackson.core.JsonParser jsonParser;

    private MonitoringJacksonJsonParserImpl(com.fasterxml.jackson.core.JsonParser jsonParser) {
        this.jsonParser = jsonParser;
    }

    public MonitoringJacksonJsonParserImpl(ByteBuf bytes) {
        this(JacksonUtils.newJsonParser(bytes));
    }

    void read(MetricJsonConsumer consumer) throws IOException {
        nextToken(JsonToken.START_OBJECT);
        while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
            String name = jsonParser.getCurrentName();
            switch (name) {
                case "ts":
                    if (!consumer.onCommonTs(readTs())) {
                        return;
                    }
                    break;
                case "labels":
                    if (!consumer.onCommonLabels(readStringStringMap())) {
                        return;
                    }
                    break;
                case "metrics":
                    if (consumer.skipMetrics()) {
                        jsonParser.skipChildren();
                    } else {
                        consumeMetrics(consumer);
                    }
                    break;
                default:
                    throw locException("Invalid attribute: " + StringUtils.wrap(name, '"'));
            }
        }
        checkToken(JsonToken.END_OBJECT);
    }

    private void consumeMetrics(MetricJsonConsumer consumer) throws IOException {
        nextToken(JsonToken.START_ARRAY);
        while (jsonParser.nextToken() != JsonToken.END_ARRAY) {
            consumer.onMetric(readMetricFromOpenObject());
        }
        checkToken(JsonToken.END_ARRAY);
    }

    private void nextToken(JsonToken expected) throws IOException {
        JsonToken jsonToken = jsonParser.nextToken();
        if (jsonToken != expected) {
            throw locException("expected: " + expected + ", got: " + wrapToken(jsonToken));
        }
    }

    private void checkToken(JsonToken expected) throws IOException {
        if (!jsonParser.hasToken(expected)) {
            throw locException("expected: " + expected + ", got: " + wrapToken(jsonParser.getCurrentToken()));
        }
    }

    @Nonnull
    private String nextString() throws IOException {
        JsonToken jsonToken = jsonParser.nextToken();
        if (jsonToken == JsonToken.VALUE_STRING) {
            return jsonParser.getText();
        }
        throw locException("Unexpected token: " + wrapToken(jsonToken));
    }

    @Nonnull
    private String readMetricName() throws IOException {
        JsonToken jsonToken = jsonParser.nextToken();
        if (jsonToken == JsonToken.VALUE_STRING) {
            String text = jsonParser.getText();
            if (StringUtils.isBlank(text)) {
                throw locException("Metric name cannot be blank");
            }

            return text;
        }

        throw locException("Unexpected token for metric name: " + wrapToken(jsonToken));
    }

    private HashMap<String, String> readStringStringMap() throws IOException {
        HashMap<String, String> r = new HashMap<>();

        nextToken(JsonToken.START_OBJECT);
        while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
            String name = jsonParser.getCurrentName();
            String value = nextString();
            r.put(name, value);
        }
        checkToken(JsonToken.END_OBJECT);

        return r;
    }

    private long readTs() throws IOException {
        JsonToken next = jsonParser.nextToken();
        if (next == JsonToken.VALUE_NUMBER_INT || next == JsonToken.VALUE_NUMBER_FLOAT) {
            return jsonParser.getLongValue();
        }

        if (next == JsonToken.VALUE_STRING) {
            return InstantUtils.parseToMillis(jsonParser.getText());
        }

        throw new IllegalArgumentException("Cannot parse timestamp from: " + wrapToken(next));
    }

    private double readValue() throws IOException {
        JsonToken next = jsonParser.nextToken();
        double value;
        if (next == JsonToken.VALUE_NUMBER_FLOAT || next == JsonToken.VALUE_NUMBER_INT) {
            value = jsonParser.getValueAsDouble();
        } else if (next == JsonToken.VALUE_STRING) {
            try {
                value = Double.parseDouble(jsonParser.getText());
            } catch (NumberFormatException e) {
                throw locException("Expecting number for value, got: " + next);
            }
        } else {
            throw locException("Expecting number for value, got: " + wrapToken(next));
        }
        return value;
    }

    private TimeSeries readTimeSeries() throws IOException {
        TimeSeries r = TimeSeries.empty();
        long prevTsMillis = 0;

        nextToken(JsonToken.START_ARRAY);
        while (jsonParser.nextToken() != JsonToken.END_ARRAY) {
            checkToken(JsonToken.START_OBJECT);

            long tsMillis = 0;
            double value = Double.NaN;

            while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
                String name = jsonParser.getCurrentName();
                switch (name) {
                    case "ts":
                        tsMillis = readTs();
                        break;
                    case "value":
                        value = readValue();
                        break;
                    default:
                        throw locException("Unknown timeseries attribute: " + StringUtils.wrap(name, '"'));
                }
            }

            if (tsMillis == 0) {
                throw locException("\"ts\" is not specified");
            }

            if (prevTsMillis > tsMillis) {
                throw locException("Timeseries with out of timestamps order");
            }

            if (Double.isNaN(value)) {
                throw locException("\"value\" is not specified");
            }

            r = r.addDouble(tsMillis, value);

            prevTsMillis = tsMillis;
            checkToken(JsonToken.END_OBJECT);
        }
        checkToken(JsonToken.END_ARRAY);

        return r;
    }

    private MetricJson readMetricFromOpenObject() throws IOException {
        MetricJson r = new MetricJson();

        checkToken(JsonToken.START_OBJECT);
        while (jsonParser.nextToken() != JsonToken.END_OBJECT) {
            String name = jsonParser.getCurrentName();
            switch (name) {
                case "name":
                    r.setName(readMetricName());
                    break;
                case "labels":
                    r.setLabels(readStringStringMap());
                    break;
                case "ts":
                    r.setTs(readTs());
                    break;
                case "value":
                    r.setValue(readValue());
                    break;
                case "type":
                    r.setType(readType());
                    break;
                case "timeseries":
                    r.setTimeSeries(readTimeSeries());
                    break;
                default:
                    throw locException("Unknown metric attribute: " + StringUtils.wrap(name, '"'));
            }
        }
        checkToken(JsonToken.END_OBJECT);

        return r;
    }

    private MetricType readType() throws IOException {
        String str = jsonParser.nextTextValue();
        if (StringUtils.isEmpty(str)) {
            return MetricType.DGAUGE;
        }
        switch (str.toUpperCase()) {
            case "DGAUGE": return MetricType.DGAUGE;
            case "IGAUGE": return MetricType.IGAUGE;
            case "COUNTER": return MetricType.COUNTER;
            case "RATE": return MetricType.RATE;
            default:
                throw locException("Unknown metric type: '" + str + '\'');
        }
    }

    private JsonParseException locException(String message) {
        return new JsonParseException(null, message + " at " + jsonParser.getCurrentLocation());
    }

    private String wrapToken(JsonToken token) {
        return StringUtils.wrap(token.asString(), '"');
    }
}
