package ru.yandex.solomon.metrics.parser.monitoringJson;

import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.buffer.ByteBuf;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.ParseException;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.monlib.metrics.labels.validate.StrictValidator;
import ru.yandex.monlib.metrics.labels.validate.TooManyLabelsException;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.metrics.parser.MetricConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.metrics.parser.json.JsonParser;
import ru.yandex.solomon.metrics.parser.json.MetricCommonData;
import ru.yandex.solomon.metrics.parser.json.MetricJson;
import ru.yandex.solomon.util.InvalidMetricNameException;
import ru.yandex.solomon.util.MetricNameHelper;

/**
 * This is copy of default Solomon JSON format parser for external monitoring
 *
 * @see ru.yandex.solomon.metrics.parser.json.TreeParserJson
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class TreeParserMonitoringJson implements TreeParser {

    private final LabelAllocator labelAllocator;
    private final String metricNameLabel;

    public TreeParserMonitoringJson(LabelAllocator labelAllocator, String metricNameLabel) {
        this.labelAllocator = labelAllocator;
        this.metricNameLabel = metricNameLabel;

        if (StringUtils.isBlank(metricNameLabel)) {
            throw new IllegalStateException("Metric name doesn't supported temporally");
        }
    }

    @Override
    public void parseAndProcess(
        Labels commonLabels,
        ByteBuf bytes,
        MetricConsumer metricConsumer,
        ErrorListener errorListener,
        FormatListener formatListener,
        boolean onlyNewFormatWrites)
    {
        JsonParser jsonParser = MonitoringJsonParser.I;

        bytes.markReaderIndex();
        MetricCommonData commonData = jsonParser.getCommonData(bytes);
        bytes.resetReaderIndex();

        LabelsBuilder labelsBuilder = new LabelsBuilder(Labels.MAX_LABELS_COUNT);
        labelsBuilder.addAll(commonLabels);
        addAllLabels(labelsBuilder, "", commonData.commonLabels, true, errorListener);

        final Labels commonLabelsFinal = labelsBuilder.build();
        final long globalTs = commonData.globalTs;

        jsonParser.forEachMetric(bytes,
            metric -> processMetric(labelsBuilder, commonLabelsFinal, metricConsumer, errorListener,
                formatListener, globalTs, metric)
        );
    }

    /**
     * @return count of added labels
     */
    private int addAllLabels(
        LabelsBuilder builder,
        String metricName,
        Map<String, String> labels,
        boolean commonLabels,
        ErrorListener errorListener)
    {
        if (!commonLabels && metricName.isEmpty()) {
            errorListener.invalidMetric(InvalidMetricReason.INVALID_SENSOR_NAME);
            throw new ParseException("Metric name cannot be empty");
        }

        int addedCount = 0;
        for (Map.Entry<String, String> e : labels.entrySet()) {
            final String name = e.getKey();
            final String value = e.getValue();

            if (LabelKeys.PROJECT.equals(name) ||
                LabelKeys.CLUSTER.equals(name) ||
                LabelKeys.SERVICE.equals(name) ||
                LabelKeys.NAME.equals(name))
            {
                errorListener.invalidMetric(InvalidMetricReason.INVALID_LABEL_NAME);
                throw new ParseException("Forbidden label key: " + name);
            }

            addLabel(builder, errorListener, name, value);
            addedCount++;
        }

        if (!commonLabels) {
            final boolean canAdd;

            try {
                canAdd =
                    MetricNameHelper.canAddMetricNameLabel(builder, metricNameLabel, metricName);
            } catch (InvalidMetricNameException e) {
                throw new ParseException("Cannot add metric name");
            }

            if (canAdd) {
                addLabel(builder, errorListener, metricNameLabel, metricName);
                addedCount++;
            }
        }

        return addedCount;
    }

    private void addLabel(
        LabelsBuilder builder,
        ErrorListener errorListener,
        String name,
        String value)
    {
        // We must validate using StrictValidator and default label validator, because
        // it's not enough to validate using StrictValidator to cover LabelValidator cases
        if (!StrictValidator.SELF.isKeyValid(name) || !LabelValidator.isValidName(name)) {
            errorListener.invalidMetric(InvalidMetricReason.INVALID_LABEL_NAME);
            throw new ParseException("Invalid label key: \"" + name + '"');
        }

        if (!StrictValidator.SELF.isValueValid(value) || !LabelValidator.isValidValue(value)) {
            errorListener.invalidMetric(InvalidMetricReason.INVALID_LABEL_VALUE);
            throw new ParseException("Invalid label value: \"" + value + '"');
        }

        try {
            builder.add(labelAllocator.alloc(name, value));
        } catch (TooManyLabelsException ex) {
            errorListener.invalidMetric(InvalidMetricReason.TOO_MANY_LABELS);
            throw new ParseException("Too many labels allocated, expected: " + Labels.MAX_LABELS_COUNT);
        }
    }

    private void processMetric(
        LabelsBuilder labelsBuilder,
        Labels commonLabels,
        MetricConsumer metricConsumer,
        ErrorListener errorListener,
        FormatListener formatListener,
        long globalTs,
        MetricJson metric)
    {
        labelsBuilder.clear();
        labelsBuilder.addAll(commonLabels);

        boolean isNewFormat = !metric.getName().isEmpty();

        formatListener.register(isNewFormat);

        int count = addAllLabels(labelsBuilder, metric.getName(), metric.getLabels(), false, errorListener);
        if (count == 0) {
            errorListener.invalidMetric(InvalidMetricReason.EMPTY_LABELS);
            throw new ParseException("No labels found");
        }

        // check that lately we can add 3 more labels for project, cluster, service
        if (!LabelsValidator.isCountValid(labelsBuilder.size() + 3)) {
            errorListener.invalidMetric(InvalidMetricReason.TOO_MANY_LABELS);
            throw new ParseException("Too many labels, expected " + (Labels.MAX_LABELS_COUNT - 3));
        }

        MetricType type = metric.getType();
        if (type == MetricType.UNKNOWN) {
            type = MetricType.DGAUGE;
        }

        long instantMillis = metric.getTs();
        if (instantMillis == 0) {
            instantMillis = globalTs;
        }

        TimeSeries timeSeries = metric.getTimeSeries();

        metricConsumer.onMetricBegin(type, labelsBuilder.build(), false);
        if (timeSeries != null) {
            metricConsumer.onTimeSeries(timeSeries);
        } else {
            metricConsumer.onPoint(instantMillis, metric.getValue());
        }
    }
}
