package ru.yandex.solomon.metrics.parser.protobuf;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.ByteString;
import com.google.protobuf.CodedInputStream;
import com.google.protobuf.ExtensionRegistryLite;
import com.google.protobuf.WireFormat;
import io.netty.buffer.ByteBuf;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monitoring.api.v3.WriteMetricsDataRequest;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.ParseException;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.monlib.metrics.labels.validate.TooManyLabelsException;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.metrics.parser.MetricConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.util.InvalidMetricNameException;
import ru.yandex.solomon.util.MetricNameHelper;
import ru.yandex.solomon.util.protobuf.ByteStrings;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class TreeParserProtobuf implements TreeParser {

    private static final int COMMON_DATA_FIELD = 1;
    private static final int METRIC_DATA_FIELD = 2;

    private final LabelAllocator labelAllocator;
    private final String metricNameLabel;

    public TreeParserProtobuf(LabelAllocator labelAllocator, String metricNameLabel) {
        this.labelAllocator = labelAllocator;
        this.metricNameLabel = metricNameLabel;
    }

    @Override
    public void parseAndProcess(
            Labels commonLabels,
            ByteBuf bytes,
            MetricConsumer metricConsumer,
            ErrorListener errorListener,
            FormatListener formatListener,
            boolean onlyNewFormatWrites)
    {
        ByteString byteString = ByteStrings.fromByteBuf(bytes);
        // read common data
        bytes.markReaderIndex();
        WriteMetricsDataRequest.CommonData commonData = parseCommonData(byteString);
        LabelsBuilder labelsBuilder = new LabelsBuilder(Labels.MAX_LABELS_COUNT);
        labelsBuilder.addAll(commonLabels);
        long commonMillis = 0;
        if (commonData != null) {
            addAllLabels(labelsBuilder, "", commonData.getLabelsMap(), true, onlyNewFormatWrites);
            commonMillis = TimeUnit.SECONDS.toMillis(commonData.getTimestamp().getSeconds());
        }
        Labels commonLabelsFinal = labelsBuilder.build();
        bytes.resetReaderIndex();

        // read and process metrics
        try {
            CodedInputStream in = CodedInputStream.newInstance(byteString.newInput());
            while (!in.isAtEnd()) {
                final int tag = in.readTag();
                if (WireFormat.getTagFieldNumber(tag) == METRIC_DATA_FIELD) {
                    var metric = in.readMessage(WriteMetricsDataRequest.Metric.parser(), ExtensionRegistryLite.getEmptyRegistry());
                    process(metric, formatListener, labelsBuilder, commonLabelsFinal, onlyNewFormatWrites, metricConsumer, commonMillis);
                } else {
                    // skip unknown field
                    in.skipField(tag);
                }
            }
        } catch (IOException e) {
            throw new ParseException("Can't parse metrics data", e);
        }
    }

    private WriteMetricsDataRequest.CommonData parseCommonData(ByteString byteString) {
        try {
            CodedInputStream in = CodedInputStream.newInstance(byteString.newInput());
            while (!in.isAtEnd()) {
                final int tag = in.readTag();
                if (WireFormat.getTagFieldNumber(tag) == COMMON_DATA_FIELD) {
                    return in.readMessage(WriteMetricsDataRequest.CommonData.parser(), ExtensionRegistryLite.getEmptyRegistry());
                } else {
                    // skip unknown field
                    in.skipField(tag);
                }
            }
        } catch (IOException e) {
            throw new ParseException("Can't parse common data", e);
        }
        return null;
    }

    private void process(
            WriteMetricsDataRequest.Metric metric,
            FormatListener formatListener,
            LabelsBuilder labelsBuilder,
            Labels commonLabels,
            boolean onlyNewFormatWrites,
            MetricConsumer metricConsumer,
            long commonMillis)
    {
        labelsBuilder.clear();
        labelsBuilder.addAll(commonLabels);

        boolean isNewFormat = !metric.getName().isEmpty();
        formatListener.register(isNewFormat);

        int count = addAllLabels(labelsBuilder, metric.getName(), metric.getLabelsMap(), false, onlyNewFormatWrites);
        if (count == 0 && commonLabels.isEmpty()) {
            throw new ParseException("Metric has empty labels");
        }

        // check that lately we can add 3 more labels for project, cluster, service
        if (!LabelsValidator.isCountValid(labelsBuilder.size() + 3)) {
            throw new ParseException("Too many labels for metric");
        }

        long instantMillis = TimeUnit.SECONDS.toMillis(metric.getTimestamp().getSeconds());
        if (instantMillis == 0) {
            instantMillis = commonMillis;
        }

        var metricType = mapType(metric.getType());
        if (metricType == null) {
            throw new ParseException("Invalid metric type: " + metric.getType());
        }
        metricConsumer.onMetricBegin(metricType, labelsBuilder.build(), false);
        if (metric.hasDoubleValue()) {
            metricConsumer.onPoint(instantMillis, metric.getDoubleValue());
        } else if (metric.hasIntValue()) {
            metricConsumer.onPoint(instantMillis, metric.getIntValue());
        } else {
            metricConsumer.onTimeSeries(parseTimeSeries(metric, commonMillis));
        }
    }

    private MetricType mapType(ru.yandex.monitoring.api.v3.MetricType type) {
        return switch (type) {
            case DGAUGE -> MetricType.DGAUGE;
            case IGAUGE -> MetricType.IGAUGE;
            case COUNTER -> MetricType.COUNTER;
            case RATE -> MetricType.RATE;
            default -> null;
        };
    }

    private TimeSeries parseTimeSeries(WriteMetricsDataRequest.Metric metric, long commonTsMillis) {
        if (metric.hasInt64Values()) {
            TimeSeries timeSeries = TimeSeries.newLong(metric.getInt64Values().getValuesCount());
            for (int i = 0; i < metric.getInt64Values().getValuesCount(); i++) {
                long millis = metric.getTimestampsList().size() > i
                        ? TimeUnit.SECONDS.toMillis(metric.getTimestamps(i).getSeconds())
                        : commonTsMillis;
                timeSeries = timeSeries.addLong(millis, metric.getInt64Values().getValues(i));
            }
            return timeSeries;
        } else if (metric.hasDoubleValues()) {
            TimeSeries timeSeries = TimeSeries.newDouble(metric.getDoubleValues().getValuesCount());
            for (int i = 0; i < metric.getDoubleValues().getValuesCount(); i++) {
                long millis = metric.getTimestampsList().size() > i
                        ? TimeUnit.SECONDS.toMillis(metric.getTimestamps(i).getSeconds())
                        : commonTsMillis;
                timeSeries = timeSeries.addDouble(millis, metric.getDoubleValues().getValues(i));
            }
            return timeSeries;
        }
        return TimeSeries.empty();
    }

    private int addAllLabels(
            LabelsBuilder builder,
            String metricName,
            Map<String, String> labels,
            boolean commonLabels,
            boolean onlyNewFormatWrites)
    {
        if (onlyNewFormatWrites && !commonLabels && metricName.isEmpty()) {
            throw new ParseException("Invalid metric name");
        }

        int addedCount = 0;
        for (Map.Entry<String, String> e : labels.entrySet()) {
            final String name = e.getKey();
            final String value = e.getValue();

            if (LabelKeys.PROJECT.equals(name) ||
                    LabelKeys.CLUSTER.equals(name) ||
                    LabelKeys.SERVICE.equals(name)) {
                if (commonLabels) {
                    // skip labels of shard key if any
                    continue;
                } else {
                    throw new ParseException("Invalid label name: " + name);
                }
            }

            if (StringUtils.isEmpty(value)) {
                if (LabelKeys.HOST.equals(name)) {
                    // host="" - is a command to remove host label
                    // see https://st.yandex-team.ru/SOLOMON-136
                    builder.remove(name);
                    continue;
                }

                // treat empty value as no label at all
                continue;
            }

            addLabel(builder, name, value);
            addedCount++;
        }


        if (!commonLabels) {
            final boolean canAdd;

            try {
                canAdd = MetricNameHelper.canAddMetricNameLabel(builder, metricNameLabel, metricName);
            } catch (InvalidMetricNameException e) {
                throw new ParseException("invalid metric name", e);
            }

            if (canAdd) {
                addLabel(builder, metricNameLabel, metricName);
            }
        }

        return addedCount;
    }

    private void addLabel(
            LabelsBuilder builder,
            String name,
            String value)
    {
        if (!LabelValidator.isValidName(name)) {
            throw new ParseException("Invalid label name: " + name);
        }

        if (!LabelValidator.isValidValue(value)) {
            throw new ParseException("Invalid label value: " + value);
        }

        try {
            builder.add(labelAllocator.alloc(name, value));
        } catch (TooManyLabelsException ex) {
            throw new ParseException("Too many labels for metric");
        }
    }
}
