package ru.yandex.solomon.metrics.parser.spack;

import java.nio.ByteBuffer;
import java.util.Map;

import io.netty.buffer.ByteBuf;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2LongMap;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.SpackException;
import ru.yandex.monlib.metrics.encode.spack.StringPool;
import ru.yandex.monlib.metrics.encode.spack.compression.DecodeStream;
import ru.yandex.monlib.metrics.encode.spack.format.MetricFlags;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.encode.spack.format.MetricValuesType;
import ru.yandex.monlib.metrics.encode.spack.format.SpackHeader;
import ru.yandex.monlib.metrics.encode.spack.format.SpackVersion;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.monlib.metrics.histogram.ExplicitHistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.monlib.metrics.labels.validate.TooManyLabelsException;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.LabelValidator;
import ru.yandex.solomon.metrics.parser.HistogramUtils;
import ru.yandex.solomon.metrics.parser.MetricConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser.ErrorListener;
import ru.yandex.solomon.metrics.parser.TreeParser.FormatListener;
import ru.yandex.solomon.metrics.parser.TreeParser.InvalidMetricReason;
import ru.yandex.solomon.util.InvalidMetricNameException;
import ru.yandex.solomon.util.MetricNameHelper;
import ru.yandex.solomon.util.collection.RecyclableLong2ObjectMap;


/**
 * @author Sergey Polovko
 */
public class SpackParserImpl {

    private final DecodeStream in;
    private final SpackHeader header;
    private final StringPool labelNames;
    private final StringPool labelValues;
    private final long commonTsMillis;
    private final LabelAllocator labelAllocator;
    private final String metricNameLabel;
    private final boolean onlyNewFormatWrites;
    private final boolean hasInvalidLabels;

    public SpackParserImpl(
        ByteBuf in,
        LabelAllocator labelAllocator,
        String metricNameLabel,
        boolean onlyNewFormatWrites)
    {
        ByteBuffer buffer = in.nioBuffer();

        this.header = SpackHeader.readFrom(buffer);

        if (onlyNewFormatWrites && header.getVersion().lt(SpackVersion.v1_2)) {
            throw new SpackException(
                    "this version " + header.getVersion() +
                    " doesn't support metric names expect >= " + SpackVersion.v1_2);
        }

        this.in = DecodeStream.create(header.getCompressionAlg(), buffer);

        int maxPoolSize = Math.max(header.getLabelNamesSize(), header.getLabelValuesSize());
        byte[] poolBuffer = new byte[maxPoolSize];

        this.in.readFully(poolBuffer, header.getLabelNamesSize());
        this.labelNames = new StringPool(poolBuffer, header.getLabelNamesSize());

        this.in.readFully(poolBuffer, header.getLabelValuesSize());
        this.labelValues = new StringPool(poolBuffer, header.getLabelValuesSize());

        this.commonTsMillis = readTsMillis();
        this.labelAllocator = labelAllocator;
        this.metricNameLabel = metricNameLabel;
        this.onlyNewFormatWrites = onlyNewFormatWrites;
        this.hasInvalidLabels = !(isAllLabelNamesValid() && isAllLabelValuesValid());
    }

    private boolean isAllLabelNamesValid() {
        for (int index = 0; index < labelNames.size(); index++) {
            String name = labelNames.get(index);
            if (!LabelValidator.isValidName(name)) {
                return false;
            }
        }

        return true;
    }

    private boolean isAllLabelValuesValid() {
        for (int index = 0; index < labelValues.size(); index++) {
            String value = labelValues.get(index);
            if (!LabelValidator.isValidValue(value)) {
                return false;
            }
        }
        return true;
    }

    public void forEachMetric(
        Labels commonLabels,
        MetricConsumer metricConsumer,
        ErrorListener errorListener,
        FormatListener formatListener)
    {
        var labelCache = RecyclableLong2ObjectMap.<Label>newInstance();
        try {
            forEachMetricInternal(commonLabels, metricConsumer, errorListener, formatListener, labelCache);
        } finally {
            labelCache.recycle();
        }
    }

    private void forEachMetricInternal(
        Labels commonLabels,
        MetricConsumer metricConsumer,
        ErrorListener errorListener,
        FormatListener formatListener,
        Long2ObjectOpenHashMap<Label> labelCache)
    {
        LabelsBuilder labelsBuilder = new LabelsBuilder(Labels.MAX_LABELS_COUNT);
        labelsBuilder.addAll(commonLabels);
        readLabels(true, "", labelsBuilder, labelCache); // labels from the document must override given commonLabels

        final Labels commonLabelsFinal = labelsBuilder.build();

        metricConsumer.ensureCapacity(header.getMetricCount());
        for (int i = 0; i < header.getMetricCount(); i++) {
            labelsBuilder.clear();
            labelsBuilder.addAll(commonLabelsFinal);

            // (1) types byte
            final byte typesByte = in.readByte();
            final MetricType metricType = MetricTypes.metricType(typesByte);
            if (metricType == MetricType.UNKNOWN) {
                throw new SpackException("unknown metric type #" + i);
            }
            final MetricValuesType metricValuesType = MetricTypes.valuesType(typesByte);

            // (2) flags byte
            final byte flagsByte = in.readByte();
            final boolean memOnly = MetricFlags.isMemOnly(flagsByte);

            // (3) name
            final String metricName = readName(formatListener);

            // (4) labels
            int labelsCount = readLabels(false, metricName, labelsBuilder, labelCache);
            if (labelsCount == 0) {
                throw new SpackException("some metric has no labels");
            }

            // check that lately we can add 3 more labels for project, cluster, service
            if (!LabelsValidator.isCountValid(labelsBuilder.size() + 3)) {
                // TODO: throw exception after checking scale of disaster
                errorListener.invalidMetric(InvalidMetricReason.TOO_MANY_LABELS);
                return;
            }

            // (4) values
            switch (metricValuesType) {
                case ONE_WITHOUT_TS: {
                    processValueOne(metricType, labelsBuilder, memOnly, commonTsMillis, metricConsumer);
                    break;
                }

                case ONE_WITH_TS: {
                    long ts = readTsMillisOrUseCommon();
                    processValueOne(metricType, labelsBuilder, memOnly, ts, metricConsumer);
                    break;
                }

                case MANY_WITH_TS: {
                    processValueMany(metricType, labelsBuilder, memOnly, metricConsumer);
                    break;
                }

                case NONE:
                    // skip metrics without values
                    continue;

                default:
                    throw new SpackException("invalid metric value type: " + metricValuesType);
            }
        }
    }

    private String readName(FormatListener formatListener) {
        if (header.getVersion().lt(SpackVersion.v1_2)) {
            formatListener.register(false);
            return "";
        }

        formatListener.register(true);
        int metricNameIdx = in.readVarint32();
        return labelValues.get(metricNameIdx);
    }

    private void processValueOne(MetricType type, LabelsBuilder labels, boolean memOnly, long ts, MetricConsumer consumer) {
        switch (type) {
            case DGAUGE: {
                consumer.onMetricBegin(type, labels.build(), memOnly);
                consumer.onPoint(ts, in.readDoubleLe());
                break;
            }
            case RATE:
            case IGAUGE:
            case COUNTER: {
                consumer.onMetricBegin(type, labels.build(), memOnly);
                consumer.onPoint(ts, in.readLongLe());
                break;
            }
            case HIST_RATE:
            case HIST: {
                processHistogramOne(type, labels, memOnly, ts, consumer);
                break;
            }
            default:
                throw new SpackException("Unsupported metric type: " + type);
        }
    }

    private void processHistogramOne(MetricType type, LabelsBuilder labels, boolean memOnly, long ts, MetricConsumer consumer) {
        HistogramSnapshot snapshot = readHistogram();
        if (snapshot.count() == 0) {
            return;
        }

        MetricType bucketType = HistogramUtils.histogramBucketType(type);
        Object2LongMap<Label> valuePerBucket = HistogramUtils.splitBuckets(snapshot, labelAllocator);
        consumer.ensureCapacity(valuePerBucket.size());
        for (Object2LongMap.Entry<Label> entry : valuePerBucket.object2LongEntrySet()) {
            Label bin = entry.getKey();
            long value = entry.getLongValue();
            consumer.onMetricBegin(bucketType, labels.add(bin).build(), memOnly);
            consumer.onPoint(ts, value);
        }
    }

    private void processValueMany(MetricType type, LabelsBuilder labels, boolean memOnly, MetricConsumer consumer) {
        switch (type) {
            case DGAUGE: {
                TimeSeries timeSeries = readDoubleTimeSeries(labels);
                consumer.onMetricBegin(type, labels.build(), memOnly);
                consumer.onTimeSeries(timeSeries);
                break;
            }
            case COUNTER:
            case IGAUGE:
            case RATE: {
                TimeSeries timeSeries = readLongTimeSeries(labels);
                consumer.onMetricBegin(type, labels.build(), memOnly);
                consumer.onTimeSeries(timeSeries);
                break;
            }
            case HIST_RATE:
            case HIST:
                processHistogramMany(type, labels, memOnly, consumer);
                break;
            default:
                throw new SpackException("Unsupported metric type: " + type);
        }
    }

    private void processHistogramMany(MetricType type, LabelsBuilder labels, boolean memOnly, MetricConsumer consumer) {
        final int count = in.readVarint32();
        if (count == 0) {
            return;
        }

        TimeSeries timeSeries = TimeSeries.newHistogram(count);
        for (int index = 0; index < count; index++) {
            long ts = readTsMillisOrUseCommon();
            HistogramSnapshot snapshot = readHistogram();
            timeSeries = timeSeries.addHistogram(ts, snapshot);
        }

        MetricType bucketType = HistogramUtils.histogramBucketType(type);
        Map<Label, TimeSeries> bucketTimeSeries = HistogramUtils.splitBuckets(timeSeries, labelAllocator);
        consumer.ensureCapacity(bucketTimeSeries.size());
        for (Map.Entry<Label, TimeSeries> entry : bucketTimeSeries.entrySet()) {
            consumer.onMetricBegin(bucketType, labels.add(entry.getKey()).build(), memOnly);
            consumer.onTimeSeries(entry.getValue());
        }
    }

    private HistogramSnapshot readHistogram() {
        int count = in.readVarint32();

        double[] bounds = new double[count];
        if (header.getVersion() == SpackVersion.v1_0) {
            // old format (bound as long)
            for (int index = 0; index < count; index++) {
                long bound = in.readLongLe();
                bounds[index] = (bound == Long.MAX_VALUE)
                    ? Histograms.INF_BOUND
                    : (double) bound;
            }
        } else {
            // new format (bound as double)
            for (int index = 0; index < count; index++) {
                bounds[index] = in.readDoubleLe();
            }
        }

        long[] buckets = new long[count];
        for (int index = 0; index < count; index++) {
            buckets[index] = in.readLongLe();
        }
        try {
            return new ExplicitHistogramSnapshot(bounds, buckets);
        } catch (IllegalArgumentException e) {
            throw new SpackException(e.getMessage());
        }
    }

    private TimeSeries readDoubleTimeSeries(LabelsBuilder labels) {
        final int count = in.readVarint32();
        TimeSeries timeSeries = TimeSeries.newDouble(count);

        long prevTs = 0;
        for (int j = 0; j < count; j++) {
            final long ts = readTsMillisOrUseCommon();
            final double value = in.readDoubleLe();

            if (prevTs > ts) {
                throw new SpackException("values has timestamps that are out of order, metric: " + labels.build());
            }

            prevTs = ts;
            timeSeries = timeSeries.addDouble(ts, value);
        }
        return timeSeries;
    }

    private TimeSeries readLongTimeSeries(LabelsBuilder labels) {
        final int count = in.readVarint32();
        TimeSeries timeSeries = TimeSeries.newLong(count);

        long prevTs = 0;
        for (int j = 0; j < count; j++) {
            final long ts = readTsMillisOrUseCommon();
            final long value = in.readLongLe();

            if (prevTs > ts) {
                throw new SpackException("values has timestamps that are out of order, metric: " + labels.build());
            }

            prevTs = ts;
            timeSeries = timeSeries.addLong(ts, value);
        }
        return timeSeries;
    }

    private long readTsMillis() {
        return header.getTimePrecision() == TimePrecision.SECONDS
            ? (long) in.readIntLe() * 1000
            : in.readLongLe();
    }

    private long readTsMillisOrUseCommon() {
        long tsMillis = readTsMillis();
        if (tsMillis == 0) {
            tsMillis = commonTsMillis;
        }
        return tsMillis;
    }

    private double readValue(MetricType metricType) {
        return (metricType == MetricType.DGAUGE)
            ? in.readDoubleLe()
            : (double) in.readLongLe();
    }

    private int readLabels(boolean commonLabels, String metricName, LabelsBuilder builder, Long2ObjectMap<Label> labelCache) {
        int addedCount = 0;
        int count = in.readVarint32();

        for (int i = 0; i < count; i++) {
            final int nameIdx = in.readVarint32();
            final int valueIdx = in.readVarint32();

            final long cacheIdx = (long) nameIdx << 32 | (long) valueIdx;
            Label label = labelCache.get(cacheIdx);
            if (label != null) {
                addLabel(builder, label);
                addedCount++;
                continue;
            }

            final String name = labelNames.get(nameIdx);
            switch (name) {
                case LabelKeys.PROJECT:
                case LabelKeys.CLUSTER:
                case LabelKeys.SERVICE: {
                    if (commonLabels) {
                        // skip labels of shard key if any
                        continue;
                    } else {
                        throw new SpackException("metric has reserved label name: \'" + name + '\'');
                    }
                }
            }

            final String value = labelValues.get(valueIdx);
            if (StringUtils.isEmpty(value)) {
                if (LabelKeys.HOST.equals(name)) {
                    // host="" - is a command to remove host label
                    // see https://st.yandex-team.ru/SOLOMON-136
                    builder.remove(name);
                    continue;
                }

                // treat empty value as no label at all
                continue;
            }

            label = createLabel(name, value);
            labelCache.put(cacheIdx, label);
            addLabel(builder, label);
            addedCount++;
        }

        if (!commonLabels) {
            if (this.onlyNewFormatWrites && metricName.isEmpty()) {
                throw new SpackException("metric name is empty");
            }

            final boolean canAdd;
            try {
                canAdd = MetricNameHelper.canAddMetricNameLabel(builder, metricNameLabel, metricName);
            } catch (InvalidMetricNameException e) {
                throw new SpackException(e.getMessage());
            }

            if (canAdd) {
                addLabel(builder, createLabel(metricNameLabel, metricName));
                addedCount++;
            }
        }

        return addedCount;
    }

    private Label createLabel(String name, String value) {
        if (hasInvalidLabels) {
            if (!LabelValidator.isValidName(name)) {
                throw new SpackException("metric has invalid label name Label{name: \'" + name + "\', value: \'" + value + "\'}");
            }

            if (!LabelValidator.isValidValue(value)) {
                throw new SpackException("metric has invalid label value Label{name: \'" + name + "\', value: \'" + value + "\'}");
            }
        }

        return labelAllocator.alloc(name, value);
    }

    private void addLabel(LabelsBuilder builder, Label label) {
        try {
            builder.add(label);
        } catch (TooManyLabelsException e) {
            throw new SpackException("too many labels in metric: " + builder.build());
        }
    }

}
