package ru.yandex.solomon.metrics.parser;

import java.time.Instant;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.series.TimeSeries;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricData extends DefaultObject implements Comparable<MetricData> {
    private final Labels labels;
    private final TimeSeries timeSeries;
    private final boolean memOnly;
    private final MetricType type;

    private MetricData(Builder builder) {
        this.labels = builder.labels;
        this.timeSeries = builder.timeSeries;
        this.memOnly = builder.memOnly;
        this.type = builder.type;
    }

    public Labels getLabels() {
        return labels;
    }

    public TimeSeries getTimeSeries() {
        return timeSeries;
    }

    public boolean isMemOnly() {
        return memOnly;
    }

    public MetricType getType() {
        return type;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static MetricData of(MetricType type, Labels labels, double value) {
        return newBuilder()
                .setType(type)
                .setLabels(labels)
                .addDouble(value)
                .build();
    }

    public static MetricData of(MetricType type, Labels labels, long value) {
        return newBuilder()
                .setType(type)
                .setLabels(labels)
                .addLong(value)
                .build();
    }

    @Override
    public int compareTo(MetricData o) {
        return getLabels().toString().compareTo(o.getLabels().toString());
    }

    public static class Builder {
        private Labels labels = Labels.empty();
        private TimeSeries timeSeries = TimeSeries.empty();
        private boolean memOnly;
        private MetricType type;

        private Builder() {
        }

        public Builder setLabels(Labels labels) {
            this.labels = labels;
            return this;
        }

        public Builder addDouble(double value) {
            return addDouble(0, value);
        }

        public Builder addDouble(long ts, double value) {
            this.timeSeries = timeSeries.addDouble(ts, value);
            return this;
        }

        public Builder addLong(long value) {
            return this.addLong(0, value);
        }

        public Builder addLong(long ts, long value) {
            this.timeSeries = timeSeries.addLong(ts, value);
            return this;
        }

        public Builder addHistogram(long ts, HistogramSnapshot value) {
            this.timeSeries = timeSeries.addHistogram(ts, value);
            return this;
        }

        public Builder addDouble(String ts, double value) {
            return addDouble(Instant.parse(ts).toEpochMilli(), value);
        }

        public Builder setTimeSeries(TimeSeries timeSeries) {
            this.timeSeries = timeSeries;
            return this;
        }

        public Builder setMemOnly(boolean memOnly) {
            this.memOnly = memOnly;
            return this;
        }

        public Builder setType(MetricType type) {
            this.type = type;
            return this;
        }

        public MetricData build() {
            return new MetricData(this);
        }
    }
}
