package ru.yandex.solomon.metrics.parser.json;

import java.io.IOException;
import java.net.URL;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.google.common.io.Resources;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.histogram.ExplicitHistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.labels.shard.ShardKey;


/**
 * @author Stepan Koltsov
 */
public class JsonParserTest {

    private static final JsonParser parser = JacksonJsonParser.I;

    protected GenericJson parse(ByteBuf bytes) {
        GenericJson json = new GenericJson();
        parser.parse(bytes, json.asJsonConsumer());
        return json;
    }

    protected GenericJson parse(String json) {
        return parse(Unpooled.copiedBuffer(json, CharsetUtil.UTF_8));
    }

    @Test
    public void test1() {
        ByteBuf bytes = loadBytes("JsonParserTest-1.json");
        GenericJson json = parse(bytes.slice());
        List<MetricJson> metrics = json.getMetrics();
        Assert.assertTrue(metrics.size() > 0);
        Assert.assertEquals(Instant.parse("2014-11-10T13:14:15Z").toEpochMilli(), json.getTs());
        Assert.assertTrue(metrics.stream().allMatch(s -> MetricJson.isValidValue(s.getValue())));

        MetricCommonData common = parser.getCommonData(bytes);
        Assert.assertEquals(Instant.parse("2014-11-10T13:14:15Z").toEpochMilli(), common.globalTs);
    }

    private static ByteBuf loadBytes(String resourceName) {
        try {
            URL resource = JsonParserTest.class.getResource(resourceName);
            return Unpooled.wrappedBuffer(Resources.toByteArray(resource));
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }

    @Test
    public void tsInMetric() {
        String json = "" +
            "{\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"BytesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 1580414596,\n" +
            "            \"mode\": \"deriv\",\n" +
            "            \"ts\": \"2014-11-10T13:14:15Z\"\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        GenericJson genericJson = parse(json);
        Assert.assertEquals(1, genericJson.getMetrics().size());
        Assert.assertEquals(Instant.parse("2014-11-10T13:14:15Z").toEpochMilli(), genericJson.getMetrics().get(0).getTs());
    }

    @Test
    public void parseShardKey() {
        ShardKey shardKey = parser.parseShardKeyOrThrow(loadBytes("JsonParserTest-1.json"));
        Assert.assertEquals(new ShardKey("kiwi", "main", "sys"), shardKey);
    }

    @Test
    public void parseShardKeyAfterMetrics() {
        String json = "" +
            "{\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"BytesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 1580414596,\n" +
            "            \"mode\": \"deriv\"\n" +
            "        },\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 157858,\n" +
            "            \"mode\": \"deriv\"\n" +
            "        }\n" +
            "    ],\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"kiwi000.search.yandex.net\",\n" +
            "        \"project\": \"kiwi\",\n" +
            "        \"cluster\": \"main\",\n" +
            "        \"service\": \"sys\"\n" +
            "    }\n" +
            "}\n";
        ShardKey shardKey = parser.parseShardKeyOrThrow(Unpooled.copiedBuffer(json, CharsetUtil.UTF_8));
        Assert.assertEquals(new ShardKey("kiwi", "main", "sys"), shardKey);

        MetricCommonData commonData = parser.getCommonData(Unpooled.copiedBuffer(json, CharsetUtil.UTF_8));
        Assert.assertEquals(Instant.parse("2014-11-10T13:14:15Z").toEpochMilli(), commonData.globalTs);
        Assert.assertEquals(Map.of(
            "host", "kiwi000.search.yandex.net",
            "project", "kiwi",
            "cluster", "main",
            "service", "sys"
        ), commonData.commonLabels);
    }

    // https://st.yandex-team.ru/SOLOMON-2095
    @Test
    public void stringValue() {
        String json = "" +
            "{\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\"\n" +
            "            },\n" +
            "            \"value\": \"157858\",\n" +
            "            \"mode\": \"deriv\"\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";
        GenericJson genericJson = parse(json);
        Assert.assertEquals(157858, genericJson.getMetrics().iterator().next().getValue(), 0.0);
    }

    // https://st.yandex-team.ru/SOLOMON-2098
    @Test
    public void numericLabelValue() {
        String json = "" +
            "{\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"metrics\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\",\n" +
            "                \"disk\": 26.5\n" +
            "            },\n" +
            "            \"value\": 157858\n" +
            "        },\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\",\n" +
            "                \"disk\": 26\n" +
            "            },\n" +
            "            \"value\": 157858\n" +
            "        },\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\",\n" +
            "                \"disk\": true\n" +
            "            },\n" +
            "            \"value\": 157858\n" +
            "        },\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\",\n" +
            "                \"disk\": false\n" +
            "            },\n" +
            "            \"value\": 157858\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";
        GenericJson genericJson = parse(json);
        List<MetricJson> metrics = genericJson.getMetrics();
        Assert.assertEquals(4, metrics.size());
        Assert.assertEquals(Map.of("sensor", "MessagesWritten", "disk", "26.5"), metrics.get(0).getLabels());
        Assert.assertEquals(Map.of("sensor", "MessagesWritten", "disk", "26"), metrics.get(1).getLabels());
        Assert.assertEquals(Map.of("sensor", "MessagesWritten", "disk", "true"), metrics.get(2).getLabels());
        Assert.assertEquals(Map.of("sensor", "MessagesWritten", "disk", "false"), metrics.get(3).getLabels());

    }

    @Test
    public void timeseries() {
        String json = "" +
            "{\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\"\n" +
            "            },\n" +
            "            \"timeseries\": [\n" +
            "                {\n" +
            "                    \"ts\": \"2015-11-10T13:14:15Z\",\n" +
            "                    \"value\": 10\n" +
            "                },\n" +
            "                {\n" +
            "                    \"ts\": \"2016-11-10T13:14:15Z\",\n" +
            "                    \"value\": 20\n" +
            "                }\n" +
            "            ],\n" +
            "            \"mode\": \"deriv\"\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";
        GenericJson genericJson = parse(json);
        MetricJson metricJson = genericJson.getMetrics().get(0);
        Assert.assertTrue(Double.isNaN(metricJson.getValue()));
        TimeSeries expected = TimeSeries.newDouble(2);
        expected.addDouble(Instant.parse("2015-11-10T13:14:15Z").toEpochMilli(), 10);
        expected.addDouble(Instant.parse("2016-11-10T13:14:15Z").toEpochMilli(), 20);

        Assert.assertEquals(expected, metricJson.getTimeSeries());
    }

    @Test
    public void histograms() {
        ByteBuf bytes = loadBytes("histograms.json");
        GenericJson json = parse(bytes.slice());

        Assert.assertEquals(Collections.singletonMap("common", "label"), json.getCommonLabels());

        List<MetricJson> metrics = json.getMetrics();
        Assert.assertEquals(3, metrics.size());

        {
            MetricJson s = metrics.get(0);
            Assert.assertEquals(MetricType.HIST, s.getType());
            Assert.assertEquals(Collections.singletonMap("sensor", "readTimeMillis"), s.getLabels());
            Assert.assertEquals(
                new ExplicitHistogramSnapshot(
                    new double[]{-1, 0, 1, 2, 4, 8, Histograms.INF_BOUND },
                    new long[]{3, 1, 2, 1, 2, 4, 91 }
                ),
                s.getHistogram());
        }
        {
            MetricJson s = metrics.get(1);
            Assert.assertEquals(MetricType.HIST_RATE, s.getType());
            Assert.assertEquals(Collections.singletonMap("sensor", "writeTimeMillis"), s.getLabels());
            Assert.assertEquals(
                new ExplicitHistogramSnapshot(
                    new double[]{ 1, 5, 15, 20, 25, Histograms.INF_BOUND },
                    new long[]{ 2, 4, 10, 5, 5, 74 }
                ),
                s.getHistogram());
        }
        {
            MetricJson s = metrics.get(2);
            Assert.assertEquals(MetricType.HIST_RATE, s.getType());
            Assert.assertEquals(Collections.singletonMap("sensor", "withoutInf"), s.getLabels());
            Assert.assertEquals(
                new ExplicitHistogramSnapshot(
                    new double[]{ 1, 5, 15, 20, 25 },
                    new long[]{ 2, 4, 10, 5, 5 }
                ),
                s.getHistogram());
        }
    }
}
