package ru.yandex.solomon.metrics.parser.json;

import java.util.List;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.UnpooledByteBufAllocator;
import io.netty.util.CharsetUtil;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.Metric;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.json.MetricJsonEncoder;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static ru.yandex.monlib.metrics.MetricType.COUNTER;
import static ru.yandex.monlib.metrics.MetricType.DGAUGE;
import static ru.yandex.monlib.metrics.MetricType.RATE;

/**
 * @author Vladimir Gordiychuk
 */
public class TreeParserJsonParametrizedTest {
    private MetricRegistry root;

    @Before
    public void setUp() {
        root = new MetricRegistry();
    }

    @Test
    public void counter() {
        addMetric(COUNTER, "upTime", TimeSeries.newLong(0, 42));
        List<MetricData> result = decode(encode(0));
        assertEquals(1, result.size());
        assertEquals(MetricData.of(COUNTER, Labels.of("sensor", "upTime"), 42.0), result.get(0));
    }

    @Test
    public void counterWithTs() {
        long now = 1517423346000L;
        addMetric(COUNTER, "errorCount", TimeSeries.newLong(now, 123));
        List<MetricData> result = decode(encode(0));

        MetricData expect = MetricData.newBuilder()
                .setType(COUNTER)
                .setLabels(Labels.of("sensor", "errorCount"))
                .addDouble(now, 123)
                .build();

        assertThat(result, allOf(iterableWithSize(1), hasItem(expect)));
    }

    @Test
    public void counterWithMany() {
        long now = 1517423346000L;
        addMetric(COUNTER, "errorCount", TimeSeries.newLong(2)
                .addLong(now + 5_000, 42)
                .addLong(now + 10_000, 52)
                .addLong(now + 15_000, 132));

        MetricData expect = MetricData.newBuilder()
                .setType(COUNTER)
                .setLabels(Labels.of("sensor", "errorCount"))
                .addDouble(now + 5_000, 42)
                .addDouble(now + 10_000, 52)
                .addDouble(now + 15_000, 132)
                .build();

        List<MetricData> result = decode(encode(0));
        assertThat(result, allOf(iterableWithSize(1), hasItem(expect)));
    }

    @Test
    public void gauge() {
        addMetric(DGAUGE, "pi", TimeSeries.newDouble(0, 3.1415d));
        List<MetricData> result = decode(encode(0));
        assertThat(result,
                allOf(
                        iterableWithSize(1),
                        hasItem(MetricData.of(DGAUGE, Labels.of("sensor", "pi"), 3.1415d))
                )
        );
    }

    @Test
    public void gaugeWithTs() {
        long now = 1517423346000L;
        addMetric(DGAUGE, "pi2", TimeSeries.newDouble(now, 3.14d));
        List<MetricData> result = decode(encode(0));

        MetricData expect = MetricData.newBuilder()
                .setType(DGAUGE)
                .setLabels(Labels.of("sensor", "pi2"))
                .addDouble(now, 3.14d)
                .build();

        assertThat(result, allOf(iterableWithSize(1), hasItem(expect)));
    }

    @Test
    public void gaugeWithMany() {
        long now = 1517423346000L;
        addMetric(DGAUGE, "q1", Labels.of("name", "value"),
                TimeSeries.newDouble(3)
                        .addDouble(now + 5_0000, 412.2d)
                        .addDouble(now + 35_0000, 3)
                        .addDouble(now + 60_0000, 42)
        );

        MetricData expect = MetricData.newBuilder()
                .setType(DGAUGE)
                .setLabels(Labels.of("sensor", "q1", "name", "value"))
                .addDouble(now + 5_0000, 412.2d)
                .addDouble(now + 35_0000, 3)
                .addDouble(now + 60_0000, 42)
                .build();

        List<MetricData> result = decode(encode(0));
        assertThat(result, allOf(iterableWithSize(1), hasItem(expect)));
    }

    @Test
    public void rate() {
        addMetric(RATE,
                "requestStarted",
                Labels.of("endpoint", "/v2/sensors"),
                TimeSeries.newLong(0, 512));

        MetricData expect = MetricData.newBuilder()
                .setLabels(Labels.of("sensor", "requestStarted", "endpoint", "/v2/sensors"))
                .addDouble(512)
                .setType(MetricType.RATE)
                .build();

        List<MetricData> result = decode(encode(0));
        assertThat(result, allOf(iterableWithSize(1), hasItem(expect)));
    }

    private void addMetric(MetricType type, String name, TimeSeries timeSeries) {
        addMetric(type, name, Labels.empty(), timeSeries);
    }

    private void addMetric(MetricType type, String name, Labels labels, TimeSeries timeSeries) {
        root.putMetricIfAbsent(new MetricId(name, labels), new ConstantMetric(type, timeSeries));
    }

    private ByteBuf encode(long now) {
        ByteBufOutputStream out = new ByteBufOutputStream(UnpooledByteBufAllocator.DEFAULT.heapBuffer());
        try (MetricEncoder encoder = new MetricJsonEncoder(out)) {
            root.accept(now, encoder);
        } catch (Exception e) {
            throw new IllegalStateException("cannot encode metric", e);
        }

        return out.buffer();
    }

    private List<MetricData> decode(ByteBuf source) {
        System.out.println(source.toString(CharsetUtil.UTF_8));

        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        TreeParser.FormatListener formatListener = TreeParser.FormatListenerIgnore.I;

        MetricDataConsumer consumer = new MetricDataConsumer();
        TreeParser parser = TreeParserJson.I;
        parser.parseAndProcess(
                Labels.empty(),
                source,
                consumer,
                errorListener, formatListener, false);

        Assert.assertFalse(errorListener.anyErrors());
        return consumer.getMetrics();
    }


    private static class ConstantMetric implements Metric {
        private final MetricType type;
        private final TimeSeries timeSeries;

        public ConstantMetric(MetricType type, TimeSeries timeSeries) {
            this.type = type;
            this.timeSeries = timeSeries;
        }

        @Override
        public MetricType type() {
            return type;
        }

        @Override
        public void accept(long tsMillis, MetricConsumer consumer) {
            for (int index = 0; index < timeSeries.size(); index++) {
                if (timeSeries.isDouble()) {
                    consumer.onDouble(timeSeries.tsMillisAt(index), timeSeries.doubleAt(index));
                } else if (timeSeries.isLong()) {
                    consumer.onLong(timeSeries.tsMillisAt(index), timeSeries.longAt(index));
                } else if (timeSeries.isHistogram()) {
                    consumer.onHistogram(timeSeries.tsMillisAt(index), timeSeries.histogramAt(index));
                } else {
                    throw new UnsupportedOperationException("Not implemented");
                }
            }
        }
    }
}
