package ru.yandex.solomon.metrics.parser.json;

import java.util.List;

import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.metrics.parser.TreeParser.ErrorListener;
import ru.yandex.solomon.metrics.parser.TreeParser.InvalidMetricReason;


/**
 * @author Stepan Koltsov
 */
public class TreeParserJsonTest {

    @Test
    public void test() {
        String json = "" +
            "{\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"kiwi000.search.yandex.net\",\n" +
            "        \"project\": \"kiwi\",\n" +
            "        \"cluster\": \"main\",\n" +
            "        \"service\": \"sys\"\n" +
            "    },\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"BytesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 1580414596,\n" +
            "            \"mode\": \"deriv\"\n" +
            "        },\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 157858,\n" +
            "            \"mode\": \"deriv\"\n" +
            "        },\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"BytesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"1\"\n" +
            "            },\n" +
            "            \"value\": 1413624626,\n" +
            "            \"mode\": \"deriv\"\n" +
            "        },\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"MessagesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"1\"\n" +
            "            },\n" +
            "            \"value\": 141198.4563,\n" +
            "            \"mode\": \"deriv\"\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json);
        Assert.assertEquals(4, r.size());

        MetricData first = MetricData.newBuilder()
                .setLabels(Labels.of(
                        "host", "kiwi000.search.yandex.net",
                        "sensor", "BytesWritten",
                        "ident", "msgtest",
                        "topic", "rt3.iva--msgtest--raw",
                        "partition", "0"))
                .setType(MetricType.RATE)
                .addDouble("2014-11-10T13:14:15Z", 1580414596)
                .build();

        Assert.assertEquals(first, r.get(0));
    }

    @Test
    public void testGlobalTs() {
        String json = "" +
            "{\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"kiwi000.search.yandex.net\",\n" +
            "        \"project\": \"kiwi\",\n" +
            "        \"cluster\": \"main\",\n" +
            "        \"service\": \"sys\"\n" +
            "    },\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"BytesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 1580414596,\n" +
            "            \"mode\": \"deriv\"\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json);
        Assert.assertEquals(1, r.size());

        MetricData expected = MetricData.newBuilder()
                .setLabels(Labels.of(
                        "host", "kiwi000.search.yandex.net",
                        "sensor", "BytesWritten",
                        "ident", "msgtest",
                        "topic", "rt3.iva--msgtest--raw",
                        "partition", "0"))
                .setType(MetricType.RATE)
                .addDouble("2014-11-10T13:14:15Z", 1580414596)
                .build();

        Assert.assertEquals(expected, r.get(0));
    }

    @Test
    public void testWithoutCommons() {
        String json = "" +
            "{\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"BytesWritten\",\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 1580414596\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        MetricData expected = MetricData.newBuilder()
                .setLabels(Labels.of(
                        "sensor", "BytesWritten",
                        "ident", "msgtest",
                        "topic", "rt3.iva--msgtest--raw",
                        "partition", "0"))
                .setType(MetricType.DGAUGE)
                .addDouble(0, 1580414596)
                .build();

        List<MetricData> result = parse(json);

        Assert.assertEquals(1, result.size());
        Assert.assertEquals(expected, result.get(0));
    }

    @Test
    public void testHist() {
        String json = "" +
                "{\n" +
                "  \"sensors\": [\n" +
                "    {\n" +
                "      \"labels\": {\n" +
                "        \"sensor\": \"elapsedTime\"\n" +
                "      },\n" +
                "          \"kind\": \"HIST\",\n" +
                "      \"timeseries\": [\n" +
                "        {\n" +
                "          \"ts\": 111,\n" +

                "          \"hist\": {\n" +
                "            \"bounds\": [100],\n" +
                "            \"buckets\": [42]\n" +
                "          }\n" +
                "        }\n" +
                "      ]\n" +
                "    }\n" +
                "  ]\n" +
                "}\n";

        List<MetricData> result = parse(json);

        Assert.assertEquals(1, result.size());
        MetricData data = result.get(0);
        Assert.assertEquals(Labels.of("sensor", "elapsedTime", "bin", "100"), data.getLabels());
        Assert.assertEquals(MetricType.IGAUGE, data.getType());
        Assert.assertEquals(TimeSeries.newLong(2).addLong(111000, 42), data.getTimeSeries());
    }

    @Test
    public void tooManyLabels() {
        String json =
            "{" +
                "  \"metrics\": [" +
                "    {" +
                "      \"labels\": {" +
                "        \"a01\": \"one\"," +
                "        \"a02\": \"two\"," +
                "        \"a03\": \"three\"," +
                "        \"a04\": \"four\"," +
                "        \"a05\": \"five\"," +
                "        \"a06\": \"six\"," +
                "        \"a07\": \"seven\"," +
                "        \"a08\": \"eight\"," +
                "        \"a09\": \"nine\"," +
                "        \"a10\": \"ten\"," +
                "        \"a11\": \"eleven\"," +
                "        \"a12\": \"twelve\"," +
                "        \"a13\": \"thirteen\"," +
                "        \"a14\": \"fourteen\"" +
                "      }," +
                "      \"value\": 3.14" +
                "    }" +
                "  ]" +
                "}";
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        List<MetricData> metrics = parse(json, errorListener);
        Assert.assertTrue(metrics.isEmpty());
        Assert.assertTrue(errorListener.anyErrors());
        Assert.assertEquals(1, errorListener.invalidMetrics.get(InvalidMetricReason.TOO_MANY_LABELS));
    }

    @Test
    public void notTooManyLabels() {
        String json =
            "{" +
                "  \"metrics\": [" +
                "    {" +
                "      \"labels\": {" +
                "        \"a01\": \"one\"," +
                "        \"a02\": \"two\"," +
                "        \"a03\": \"three\"," +
                "        \"a04\": \"four\"," +
                "        \"a05\": \"five\"," +
                "        \"a06\": \"six\"," +
                "        \"a07\": \"seven\"," +
                "        \"a08\": \"eight\"," +
                "        \"a09\": \"nine\"," +
                "        \"a10\": \"ten\"," +
                "        \"a11\": \"eleven\"," +
                "        \"a12\": \"twelve\"," +
                "        \"a13\": \"thirteen\"" +
                "      }," +
                "      \"value\": 3.14" +
                "    }" +
                "  ]" +
                "}";
        List<MetricData> metrics = parse(json);
        Assert.assertFalse(metrics.isEmpty());

        MetricData metric = metrics.get(0);
        Assert.assertEquals(MetricType.DGAUGE, metric.getType());
        Assert.assertEquals(
            Labels.builder()
                .add("a01", "one")
                .add("a02", "two")
                .add("a03", "three")
                .add("a04", "four")
                .add("a05", "five")
                .add("a06", "six")
                .add("a07", "seven")
                .add("a08", "eight")
                .add("a09", "nine")
                .add("a10", "ten")
                .add("a11", "eleven")
                .add("a12", "twelve")
                .add("a13", "thirteen")
                .build(),
            metric.getLabels());
        Assert.assertEquals(TimeSeries.newDouble(0, 3.14), metric.getTimeSeries());
    }

    private List<MetricData> parse(String json) {
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        List<MetricData> result = parse(json, errorListener);
        Assert.assertFalse(errorListener.anyErrors());
        return result;
    }

    private List<MetricData> parse(String json, ErrorListener errorListener) {
        MetricDataConsumer consumer = new MetricDataConsumer();
        TreeParserJson.I.parseAndProcess(
            Labels.empty(),
            Unpooled.copiedBuffer(json, CharsetUtil.UTF_8),
            consumer,
            errorListener,
            TreeParser.FormatListenerIgnore.I, false);
        return consumer.getMetrics();
    }
}
