package ru.yandex.solomon.metrics.parser.json;

import java.util.List;

import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;


/**
 * @author Oleg Baryshnikov
 */
public class TreeParserJsonWithMetricNameTest {

    @Test
    public void testMetricNameInOldFormat() {
        String json = "" +
            "{\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"localhost\",\n" +
            "        \"project\": \"solomon\",\n" +
            "        \"cluster\": \"production\",\n" +
            "        \"service\": \"sys\"\n" +
            "    },\n" +
            "    \"ts\": \"2019-02-19T11:35:00Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"path\": \"/Filesystem/SizeB\",\n" +
            "                \"mountpoint\": \"/\"\n" +
            "            },\n" +
            "            \"value\": 17950958477922\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json, "path");
        Assert.assertEquals(1, r.size());

        MetricData first = MetricData.newBuilder()
            .setLabels(Labels.of(
                "host", "localhost",
                "path", "/Filesystem/SizeB",
                "mountpoint", "/"))
            .setType(MetricType.DGAUGE)
            .addDouble("2019-02-19T11:35:00Z", 17950958477922d)
            .build();

        Assert.assertEquals(first, r.get(0));
    }

    @Test
    public void testMetricNameInNewFormat() {
        String json = "" +
            "{\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"localhost\",\n" +
            "        \"project\": \"solomon\",\n" +
            "        \"cluster\": \"production\",\n" +
            "        \"service\": \"sys\"\n" +
            "    },\n" +
            "    \"ts\": \"2019-02-19T11:40:00Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"name\": \"/Filesystem/SizeB\",\n" +
            "            \"labels\": {\n" +
            "                \"mountpoint\": \"/\"\n" +
            "            },\n" +
            "            \"value\": 17950958477922\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json, "sensor");
        Assert.assertEquals(1, r.size());

        MetricData first = MetricData.newBuilder()
            .setLabels(Labels.of(
                "host", "localhost",
                "sensor", "/Filesystem/SizeB",
                "mountpoint", "/"))
            .setType(MetricType.DGAUGE)
            .addDouble("2019-02-19T11:40:00Z", 17950958477922d)
            .build();

        Assert.assertEquals(first, r.get(0));
    }

    @Test
    public void testMetricNameAndEmptyLabels() {
        String json = "" +
            "{\n" +
            "    \"ts\": \"2019-02-19T11:40:00Z\",\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"name\": \"/Filesystem/SizeB\",\n" +
            "            \"value\": 17950958477922\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json, "sensor");
        Assert.assertEquals(1, r.size());

        MetricData first = MetricData.newBuilder()
            .setLabels(Labels.of("sensor", "/Filesystem/SizeB"))
            .setType(MetricType.DGAUGE)
            .addDouble("2019-02-19T11:40:00Z", 17950958477922d)
            .build();

        Assert.assertEquals(first, r.get(0));
    }

    @Test
    public void testDuplicatedMetricName() {
        String json = "" +
            "{\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"localhost\",\n" +
            "        \"project\": \"solomon\",\n" +
            "        \"cluster\": \"production\",\n" +
            "        \"service\": \"sys\",\n" +
            "        \"sensor\": \"/Filesystem/SizeB\"\n" +
            "    },\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"name\": \"/Filesystem/SizeB\",\n" +
            "            \"labels\": {\n" +
            "                \"mountpoint\": \"/\"\n" +
            "            },\n" +
            "            \"value\": 17950958477922\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        parseWithError(json, "sensor", false);
    }

    @Test
    public void testUnknownMetricName() {
        String json = "" +
            "{\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"localhost\",\n" +
            "        \"project\": \"solomon\",\n" +
            "        \"cluster\": \"production\",\n" +
            "        \"service\": \"sys\"\n" +
            "    },\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"sensor\": \"/Filesystem/SizeB\",\n" +
            "                \"mountpoint\": \"/\"\n" +
            "            },\n" +
            "            \"value\": 17950958477922\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        parseWithError(json, "path", false);
    }

    @Test
    public void testOnlyNewFormatWrites() {
        String json = "" +
            "{\n" +
            "    \"commonLabels\": {\n" +
            "        \"host\": \"localhost\",\n" +
            "        \"project\": \"solomon\",\n" +
            "        \"cluster\": \"production\",\n" +
            "        \"service\": \"sys\"\n" +
            "    },\n" +
            "    \"sensors\": [\n" +
            "        {\n" +
            "            \"labels\": {\n" +
            "                \"mountpoint\": \"/\"\n" +
            "            },\n" +
            "            \"value\": 17950958477922\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        parseWithError(json, "sensor", true);
    }

    private List<MetricData> parse(String json, String metricNameLabel) {
        MetricDataConsumer consumer = new MetricDataConsumer();
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        TreeParserJson treeParser = new TreeParserJson(Labels.allocator, metricNameLabel);
        treeParser.parseAndProcess(
                Labels.empty(),
                Unpooled.copiedBuffer(json, CharsetUtil.UTF_8),
                consumer,
                errorListener,
                TreeParser.FormatListenerIgnore.I,
                false
        );
        Assert.assertFalse(errorListener.anyErrors());
        return consumer.getMetrics();
    }

    private void parseWithError(
        String json,
        String metricNameLabel,
        boolean onlyNewFormatWrites)
    {
        MetricDataConsumer consumer = new MetricDataConsumer();
        ErrorListenerCounter errorListener = new ErrorListenerCounter();

        TreeParserJson treeParser = new TreeParserJson(Labels.allocator, metricNameLabel);
        treeParser.parseAndProcess(
                Labels.empty(),
                Unpooled.copiedBuffer(json, CharsetUtil.UTF_8),
                consumer,
                errorListener,
                TreeParser.FormatListenerIgnore.I,
                onlyNewFormatWrites
        );
        Assert.assertTrue(errorListener.anyErrors());
        Assert.assertEquals(1, errorListener.invalidMetrics.get(
            TreeParser.InvalidMetricReason.INVALID_SENSOR_NAME));
    }
}
