package ru.yandex.solomon.metrics.parser.monitoringJson;

import java.util.List;

import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.ParseException;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;

/**
 * @author Oleg Baryshnikov
 */
public class TreeParserMonitoringJsonTest {

    @Test
    public void test() {
        String json =
            "{\n" +
            "    \"labels\": {\n" +
            "        \"host\": \"kiwi000.search.yandex.net\"\n" +
            "    },\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"metrics\": [\n" +
            "        {\n" +
            "            \"name\": \"BytesWritten\",\n" +
            "            \"labels\": {\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"type\": \"RATE\",\n" +
            "            \"value\": 1580414596\n" +
            "        },\n" +
            "        {\n" +
            "            \"name\": \"MessagesWritten\",\n" +
            "            \"labels\": {\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"type\": \"RATE\",\n" +
            "            \"value\": 157858\n" +
            "        },\n" +
            "        {\n" +
            "            \"name\": \"BytesWritten\",\n" +
            "            \"labels\": {\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"1\"\n" +
            "            },\n" +
            "            \"type\": \"RATE\",\n" +
            "            \"value\": 1413624626\n" +
            "        },\n" +
            "        {\n" +
            "            \"name\": \"MessagesWritten\",\n" +
            "            \"labels\": {\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"1\"\n" +
            "            },\n" +
            "            \"type\": \"RATE\",\n" +
            "            \"value\": 141198.4563\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json);
        Assert.assertEquals(4, r.size());

        MetricData first = MetricData.newBuilder()
            .setLabels(Labels.of(
                "host", "kiwi000.search.yandex.net",
                "name", "BytesWritten",
                "ident", "msgtest",
                "topic", "rt3.iva--msgtest--raw",
                "partition", "0"))
            .setType(MetricType.RATE)
            .addDouble("2014-11-10T13:14:15Z", 1580414596)
            .build();

        Assert.assertEquals(first, r.get(0));
    }

    @Test
    public void testWithoutLabels() {
        String json =
            "{" +
                "  \"metrics\": [" +
                "    {" +
                "      \"name\": \"metric\"," +
                "      \"value\": 100" +
                "    }" +
                "  ]" +
                "}";
        List<MetricData> metrics = parse(json);
        Assert.assertFalse(metrics.isEmpty());

        MetricData metric = metrics.get(0);
        Assert.assertEquals(MetricType.DGAUGE, metric.getType());
        Assert.assertEquals(
            Labels.builder()
                .add("name", "metric")
                .build(),
            metric.getLabels());
        Assert.assertEquals(TimeSeries.newDouble(0, 100), metric.getTimeSeries());
    }

    @Test
    public void testGlobalTs() {
        String json =
            "{\n" +
            "    \"labels\": {\n" +
            "        \"host\": \"kiwi000.search.yandex.net\"\n" +
            "    },\n" +
            "    \"ts\": \"2014-11-10T13:14:15Z\",\n" +
            "    \"metrics\": [\n" +
            "        {\n" +
            "            \"name\": \"BytesWritten\",\n" +
            "            \"labels\": {\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 1580414596,\n" +
            "            \"type\": \"RATE\"\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json);
        Assert.assertEquals(1, r.size());

        MetricData expected = MetricData.newBuilder()
            .setLabels(Labels.of(
                "host", "kiwi000.search.yandex.net",
                "name", "BytesWritten",
                "ident", "msgtest",
                "topic", "rt3.iva--msgtest--raw",
                "partition", "0"))
            .setType(MetricType.RATE)
            .addDouble("2014-11-10T13:14:15Z", 1580414596)
            .build();

        Assert.assertEquals(expected, r.get(0));
    }

    @Test
    public void testWithoutCommons() {
        String json =
            "{\n" +
            "    \"metrics\": [\n" +
            "        {\n" +
            "            \"name\": \"BytesWritten\",\n" +
            "            \"labels\": {\n" +
            "                \"ident\": \"msgtest\",\n" +
            "                \"topic\": \"rt3.iva--msgtest--raw\",\n" +
            "                \"partition\": \"0\"\n" +
            "            },\n" +
            "            \"value\": 1580414596\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        MetricData expected = MetricData.newBuilder()
            .setLabels(Labels.of(
                "name", "BytesWritten",
                "ident", "msgtest",
                "topic", "rt3.iva--msgtest--raw",
                "partition", "0"))
            .setType(MetricType.DGAUGE)
            .addDouble(0, 1580414596)
            .build();

        List<MetricData> result = parse(json);

        Assert.assertEquals(1, result.size());
        Assert.assertEquals(expected, result.get(0));
    }

    @Test
    public void tooManyLabels() {
        String json =
            "{" +
                "  \"metrics\": [" +
                "    {" +
                "      \"name\": \"metric\"," +
                "      \"labels\": {" +
                "        \"a01\": \"one\"," +
                "        \"a02\": \"two\"," +
                "        \"a03\": \"three\"," +
                "        \"a04\": \"four\"," +
                "        \"a05\": \"five\"," +
                "        \"a06\": \"six\"," +
                "        \"a07\": \"seven\"," +
                "        \"a08\": \"eight\"," +
                "        \"a09\": \"nine\"," +
                "        \"a10\": \"ten\"," +
                "        \"a11\": \"eleven\"," +
                "        \"a12\": \"twelve\"," +
                "        \"a13\": \"thirteen\"" +
                "      }," +
                "      \"value\": 3.14\n" +
                "    }" +
                "  ]" +
                "}";
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        try {
            List<MetricData> metrics = parse(json, errorListener);
        } catch (ParseException e) {
            Assert.assertTrue(errorListener.anyErrors());
            Assert.assertEquals(1,
                errorListener.invalidMetrics.get(TreeParser.InvalidMetricReason.TOO_MANY_LABELS));
            return;
        }

        throw new AssertionError("Expected parse error");
    }

    @Test
    public void notTooManyLabels() {
        String json =
            "{" +
                "  \"metrics\": [" +
                "    {" +
                "      \"name\": \"metric\"," +
                "      \"labels\": {" +
                "        \"a01\": \"one\"," +
                "        \"a02\": \"two\"," +
                "        \"a03\": \"three\"," +
                "        \"a04\": \"four\"," +
                "        \"a05\": \"five\"," +
                "        \"a06\": \"six\"," +
                "        \"a07\": \"seven\"," +
                "        \"a08\": \"eight\"," +
                "        \"a09\": \"nine\"," +
                "        \"a10\": \"ten\"," +
                "        \"a11\": \"eleven\"," +
                "        \"a12\": \"twelve\"" +
                "      }," +
                "      \"value\": 3.14" +
                "    }" +
                "  ]" +
                "}";
        List<MetricData> metrics = parse(json);
        Assert.assertFalse(metrics.isEmpty());

        MetricData metric = metrics.get(0);
        Assert.assertEquals(MetricType.DGAUGE, metric.getType());
        Assert.assertEquals(
            Labels.builder()
                .add("name", "metric")
                .add("a01", "one")
                .add("a02", "two")
                .add("a03", "three")
                .add("a04", "four")
                .add("a05", "five")
                .add("a06", "six")
                .add("a07", "seven")
                .add("a08", "eight")
                .add("a09", "nine")
                .add("a10", "ten")
                .add("a11", "eleven")
                .add("a12", "twelve")
                .build(),
            metric.getLabels());
        Assert.assertEquals(TimeSeries.newDouble(0, 3.14), metric.getTimeSeries());
    }

    @Test
    public void numericTs() {
        String json =
            "{\n" +
            "    \"labels\": {\n" +
            "        \"host\": \"kiwi000.search.yandex.net\"\n" +
            "    },\n" +
            "    \"ts\": 1571234567890,\n" +
            "    \"metrics\": [\n" +
            "        {\n" +
            "            \"name\": \"BytesWritten\",\n" +
            "            \"value\": 123\n" +
            "        },\n" +
            "        {\n" +
            "            \"name\": \"BytesRead\",\n" +
            "            \"ts\": 1579876543210,\n" +
            "            \"value\": 456\n" +
            "        }\n" +
            "    ]\n" +
            "}\n";

        List<MetricData> r = parse(json);
        Assert.assertEquals(2, r.size());

        MetricData s1 = MetricData.newBuilder()
            .setLabels(Labels.of("host", "kiwi000.search.yandex.net", "name", "BytesWritten"))
            .setType(MetricType.DGAUGE)
            .addDouble("2019-10-16T14:02:47.890Z", 123)
            .build();

        Assert.assertEquals(s1, r.get(0));

        MetricData s2 = MetricData.newBuilder()
            .setLabels(Labels.of("host", "kiwi000.search.yandex.net", "name", "BytesRead"))
            .setType(MetricType.DGAUGE)
            .addDouble("2020-01-24T14:35:43.210Z", 456)
            .build();

        Assert.assertEquals(s2, r.get(1));


    }

    private List<MetricData> parse(String json) {
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        List<MetricData> result = parse(json, errorListener);
        Assert.assertFalse(errorListener.anyErrors());
        return result;
    }

    private List<MetricData> parse(String json, TreeParser.ErrorListener errorListener) {
        TreeParserMonitoringJson parser =
            new TreeParserMonitoringJson(Labels.allocator, "name");

        MetricDataConsumer consumer = new MetricDataConsumer();

        parser.parseAndProcess(
            Labels.empty(),
            Unpooled.copiedBuffer(json, CharsetUtil.UTF_8),
            consumer,
            errorListener,
            TreeParser.FormatListenerIgnore.I,
            true);

        return consumer.getMetrics();
    }
}
