package ru.yandex.solomon.metrics.parser.prometheus;

import java.io.IOException;
import java.net.URL;
import java.util.List;

import com.google.common.io.Resources;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.junit.Test;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author Sergey Polovko
 */
public class RemoteWriteParserTest {

    @Test
    public void smoke() {
        List<MetricData> metrics = decode(loadResource("samples.prom"));
        assertEquals(12, metrics.size());

        for (MetricData metric : metrics) {
            // all metrics are gauge
            assertEquals(MetricType.DGAUGE, metric.getType());

            // there is no such thing as memonly metric in prometheus
            assertFalse(metric.isMemOnly());

            // all metrics have a single point
            assertEquals(1, metric.getTimeSeries().size());
            // with the same timestamp
            assertEquals(1630960403695L, metric.getTimeSeries().tsMillisAt(0));

            // all points are finite doubles
            assertTrue(metric.getTimeSeries().isDouble());
            assertTrue(Double.isFinite(metric.getTimeSeries().doubleAt(0)));

            // '__name__' label was remapped to a 'sensor' label
            assertFalse(metric.getLabels().hasKey("__name__"));
            assertTrue(metric.getLabels().hasKey("sensor"));
        }
    }

    private List<MetricData> decode(ByteBuf source) {
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        MetricDataConsumer consumer = new MetricDataConsumer();

        TreeParser parser = new RemoteWriteParser(Labels.allocator, "sensor");
        parser.parseAndProcess(
            Labels.of("project", "solomon"),
            source,
            consumer,
            errorListener, TreeParser.FormatListenerIgnore.I, false);

        assertFalse(errorListener.anyErrors());
        return consumer.getMetrics();
    }

    private static ByteBuf loadResource(String name) {
        try {
            URL resource = RemoteWriteParserTest.class.getResource(name);
            return Unpooled.wrappedBuffer(Resources.toByteArray(resource));
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }
}
