package ru.yandex.solomon.metrics.parser.protobuf;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.Timestamp;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monitoring.api.v3.DoubleValues;
import ru.yandex.monitoring.api.v3.Int64Values;
import ru.yandex.monitoring.api.v3.WriteMetricsDataRequest;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;

/**
 * @author Alexey Trushkin
 */
public class TreeParserProtobufTest {

    private static final long COMMON_TS_MILLIS = Instant.parse("2017-08-27T12:34:56Z").toEpochMilli();
    private static final long TS_MILLIS = Instant.parse("2017-08-28T12:34:56Z").toEpochMilli();

    private static final MetricData[] EXPECTED_METRICS = {
            MetricData.newBuilder()
                    .setType(MetricType.IGAUGE)
                    .setLabels(Labels.of("common", "value", "v1", "v2", "nameLabel", "name"))
                    .addLong(COMMON_TS_MILLIS, 1)
                    .build(),

            MetricData.newBuilder()
                    .setType(MetricType.DGAUGE)
                    .setLabels(Labels.of("common", "value", "v2", "v3", "nameLabel", "name"))
                    .addDouble(TS_MILLIS, 1.1)
                    .build(),

            MetricData.newBuilder()
                    .setType(MetricType.COUNTER)
                    .setLabels(Labels.of("common", "value", "v3", "v4", "nameLabel", "name"))
                    .addLong(COMMON_TS_MILLIS, 1)
                    .addLong(COMMON_TS_MILLIS, 2)
                    .build(),

            MetricData.newBuilder()
                    .setType(MetricType.DGAUGE)
                    .setLabels(Labels.of("common", "value", "v4", "v5", "nameLabel", "name"))
                    .addDouble(COMMON_TS_MILLIS, 1.1)
                    .addDouble(TS_MILLIS, 2.2)
                    .build(),

    };

    @Test
    public void parseMultiple() {
        var data = newData();
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        List<MetricData> metrics = parse(data, errorListener);
        Assert.assertEquals(4, metrics.size());
        Assert.assertFalse(errorListener.anyErrors());

        Assert.assertTrue(metrics.contains(EXPECTED_METRICS[0]));
        Assert.assertTrue(metrics.contains(EXPECTED_METRICS[1]));
        Assert.assertTrue(metrics.contains(EXPECTED_METRICS[2]));
        Assert.assertTrue(metrics.contains(EXPECTED_METRICS[3]));
    }

    private WriteMetricsDataRequest.TypedData newData() {
        return WriteMetricsDataRequest.TypedData.newBuilder()
                .setCommonData(WriteMetricsDataRequest.CommonData.newBuilder()
                        .putAllLabels(Map.of("common", "value"))
                        .setTimestamp(Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(COMMON_TS_MILLIS)).build())
                        .build())
                .addAllMetrics(List.of(
                        WriteMetricsDataRequest.Metric.newBuilder()
                                .setType(ru.yandex.monitoring.api.v3.MetricType.IGAUGE)
                                .putAllLabels(Map.of("v1", "v2", "nameLabel", "name"))
                                .setTimestamp(Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(COMMON_TS_MILLIS)).build())
                                .setIntValue(1L)
                                .build(),
                        WriteMetricsDataRequest.Metric.newBuilder()
                                .setType(ru.yandex.monitoring.api.v3.MetricType.DGAUGE)
                                .putAllLabels(Map.of("v2", "v3", "nameLabel", "name"))
                                .setTimestamp(Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(TS_MILLIS)).build())
                                .setDoubleValue(1.1)
                                .build(),
                        WriteMetricsDataRequest.Metric.newBuilder()
                                .setType(ru.yandex.monitoring.api.v3.MetricType.COUNTER)
                                .putAllLabels(Map.of("v3", "v4", "nameLabel", "name"))
                                .setInt64Values(Int64Values.newBuilder()
                                        .addAllValues(List.of(1L, 2L))
                                        .build())
                                .build(),
                        WriteMetricsDataRequest.Metric.newBuilder()
                                .setType(ru.yandex.monitoring.api.v3.MetricType.DGAUGE)
                                .putAllLabels(Map.of("v4", "v5", "nameLabel", "name"))
                                .addAllTimestamps(List.of(
                                        Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(COMMON_TS_MILLIS)).build(),
                                        Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(TS_MILLIS)).build()
                                ))
                                .setDoubleValues(DoubleValues.newBuilder()
                                        .addAllValues(List.of(1.1, 2.2))
                                        .build())
                                .build()
                ))
                .build();
    }

    private List<MetricData> parse(WriteMetricsDataRequest.TypedData data, TreeParser.ErrorListener errorListener) {
        MetricDataConsumer consumer = new MetricDataConsumer();
        TreeParserProtobuf parser = new TreeParserProtobuf(Labels.allocator, "nameLabel");
        ByteBuf byteBuf = Unpooled.wrappedBuffer(data.toByteArray());
        parser.parseAndProcess(
                Labels.of(),
                byteBuf,
                consumer,
                errorListener,
                TreeParser.FormatListenerIgnore.I,
                false);

        return consumer.getMetrics();
    }
}
