package ru.yandex.solomon.metrics.parser.spack;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.MetricFormat;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.labels.string.StringLabelAllocator;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static ru.yandex.monlib.metrics.encode.MetricEncoderFactory.createEncoder;

/**
 * @author Vladimir Gordiychuk
 */
public class SpackParserImplTest {

    private MetricRegistry registry;

    @Before
    public void setUp() {
        registry = new MetricRegistry();
    }

    @Test
    public void empty() {
        List<MetricData> result = encodeDecode(registry);
        assertEquals(List.of(), result);
    }

    @Test
    public void parseOne() {
        registry.rate("myRateSensor", Labels.of("key1", "value1", "key2", "value2")).set(42);
        List<MetricData> result = encodeDecode(registry);
        assertEquals(1, result.size());
        assertEquals(MetricData.newBuilder()
            .setType(MetricType.RATE)
            .setLabels(Labels.of("sensor", "myRateSensor", "key1", "value1", "key2", "value2"))
            .setMemOnly(false)
            .addLong(42)
            .build(), result.get(0));
    }

    @Test
    public void parseMultipleWithSameKey() {
        Labels common = Labels.of("key1", "value1", "key2", "value2");
        registry.rate("alice", common).set(10);
        registry.rate("bob", common).set(20);

        List<MetricData> result = encodeDecode(registry);
        assertEquals(2, result.size());

        List<MetricData> expected = Arrays.asList(
            MetricData.newBuilder()
                .setMemOnly(false)
                .setLabels(common.add("sensor", "alice"))
                .setType(MetricType.RATE)
                .addLong(10)
                .build(),

            MetricData.newBuilder()
                .setMemOnly(false)
                .setLabels(common.add("sensor", "bob"))
                .setType(MetricType.RATE)
                .addLong(20)
                .build());

        Collections.sort(expected);
        Collections.sort(result);
        assertEquals(expected, result);

        List<MetricData> resultTwo = encodeDecode(registry);
        Collections.sort(resultTwo);
        assertEquals(expected, resultTwo);
    }

    @Test
    public void random() {
        List<MetricData> expected = new ArrayList<>();
        Set<Labels> collision = new HashSet<>();
        for (int index = 0; index < 500; index++) {
            var labels = randomLabels();
            var expectLabels = labels.add("sensor", "test");
            if (!collision.add(expectLabels)) {
                continue;
            }

            registry.rate("test", expectLabels).set(index);
            expected.add(MetricData.newBuilder()
                .setMemOnly(false)
                .setLabels(expectLabels)
                .setType(MetricType.RATE)
                .addLong(index)
                .build());
        }

        Collections.sort(expected);
        var results = IntStream.range(0, 100)
            .parallel()
            .mapToObj(ignore -> encodeDecode(registry))
            .collect(Collectors.toList());

        for (var result : results) {
            Collections.sort(result);
            assertArrayEquals(expected.toArray(), result.toArray());
        }
    }

    private Labels randomLabels() {
        int size = ThreadLocalRandom.current().nextInt(1, 10);
        var builder = new LabelsBuilder(size);
        for (int index = 0; index < size; index++) {
            builder.add(randomLabel());
        }
        return builder.build();
    }

    private Label randomLabel() {
        String key = "key" + ThreadLocalRandom.current().nextInt(5);
        String value = "value" + ThreadLocalRandom.current().nextInt(5);
        return Labels.allocator.alloc(key, value);
    }

    private List<MetricData> encodeDecode(MetricRegistry registry) {
        var encoded = encode(registry);
        return parse(encoded);
    }

    private List<MetricData> parse(ByteBuf buffer) {
        SpackParserImpl parser = new SpackParserImpl(buffer, StringLabelAllocator.SELF, "sensor", false);
        MetricDataConsumer consumer = new MetricDataConsumer();
        parser.forEachMetric(Labels.empty(), consumer, TreeParser.ErrorListenerIgnore.I, TreeParser.FormatListenerIgnore.I);
        return consumer.getMetrics();
    }

    private ByteBuf encode(MetricRegistry registry) {
        ByteArrayOutputStream out = new ByteArrayOutputStream(8 << 10); // 8 KiB
        try (MetricEncoder encoder = createEncoder(out, MetricFormat.SPACK, CompressionAlg.LZ4)) {
            registry.supply(0, encoder);
        } catch (Exception e) {
            throw new IllegalStateException("cannot encode metrics", e);
        }

        return Unpooled.wrappedBuffer(out.toByteArray());
    }
}
