package ru.yandex.solomon.metrics.parser.spack;

import java.util.Arrays;
import java.util.List;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.monlib.metrics.Metric;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.MetricEncoder;
import ru.yandex.monlib.metrics.encode.spack.MetricSpackEncoder;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.monlib.metrics.histogram.ExplicitHistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;

import static org.hamcrest.Matchers.hasItems;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
@RunWith(Parameterized.class)
public class TreeParserSpackParametrizedTest {

    @Parameterized.Parameter
    public CompressionAlg compression;

    @Parameterized.Parameters(name = "{0}")
    public static CompressionAlg[] data() {
        return CompressionAlg.values();
    }

    @Test
    public void parseCounterHistogram() {
        MetricRegistry registry = new MetricRegistry(Labels.of("project", "solomon"));
        Histogram counter = registry.histogramCounter("q1", Histograms.explicit(10, 20, 30));

        counter.record(2);
        counter.record(1);
        counter.record(15);

        MetricData[] result = decode(encode(0, registry)).stream()
            .sorted()
            .toArray(MetricData[]::new);

        MetricData[] expected = {
            MetricData.of(MetricType.IGAUGE, Labels.of("bin", "10", "project", "solomon", "sensor", "q1"), 2),
            MetricData.of(MetricType.IGAUGE, Labels.of("bin", "20", "project", "solomon", "sensor", "q1"), 1),
            MetricData.of(MetricType.IGAUGE, Labels.of("bin", "30", "project", "solomon", "sensor", "q1"), 0),
            MetricData.of(MetricType.IGAUGE, Labels.of("bin", "inf", "project", "solomon", "sensor", "q1"), 0)
        };
        Arrays.sort(expected);

        Assert.assertArrayEquals(expected, result);
    }

    @Test
    public void parseRateHistogram() {
        MetricRegistry registry = new MetricRegistry(Labels.of("project", "solomon"));
        Histogram counter = registry.histogramRate("q2", Histograms.explicit(10, 30));

        counter.record(13);
        counter.record(42);
        counter.record(100500);

        ByteBuf source = encode(0, registry);
        List<MetricData> result = decode(source);

        MetricData bin10 = MetricData.newBuilder()
                .setLabels(Labels.of("bin", "10", "project", "solomon", "sensor", "q2"))
                .setType(MetricType.RATE)
                .addLong(0)
                .build();

        MetricData bin30 = MetricData.newBuilder()
                .setLabels(Labels.of("bin", "30", "project", "solomon", "sensor", "q2"))
                .setType(MetricType.RATE)
                .addLong(1)
                .build();

        MetricData binInf = MetricData.newBuilder()
                .setLabels(Labels.of("bin", "inf", "project", "solomon", "sensor", "q2"))
                .setType(MetricType.RATE)
                .addLong(2)
                .build();

        assertThat("2 explicit bucket and one additional for inf", result, Matchers.iterableWithSize(3));
        assertThat(result, hasItems(bin10, bin30, binInf));
    }

    @Test
    public void parseMultipleHistogram() {
        long now = 1517423346000L;
        MetricRegistry registry = new MetricRegistry(Labels.of("project", "solomon"));
        registry.putMetricIfAbsent(new MetricId("q3", Labels.empty()), new Metric() {
            @Override
            public MetricType type() {
                return MetricType.HIST;
            }

            @Override
            public void accept(long tsMillis, MetricConsumer consumer) {
                double[] bounds = {10, 20, 30, Histograms.INF_BOUND};
                consumer.onHistogram(now + 5_000,  hist(bounds, new long[]{0, 2, 1, 0}));
                consumer.onHistogram(now + 10_000, hist(bounds, new long[]{3, 0, 0, 0}));
                consumer.onHistogram(now + 15_000, hist(bounds, new long[]{1, 1, 5, 8}));
            }
        });

        ByteBuf source = encode(now, registry);
        List<MetricData> result = decode(source);

        MetricData bin10 = MetricData.newBuilder()
                .setType(MetricType.IGAUGE)
                .setLabels(Labels.of("bin", "10", "project", "solomon", "sensor", "q3"))
                .addLong(now + 5_000,  0)
                .addLong(now + 10_000, 3)
                .addLong(now + 15_000, 1)
                .build();

        MetricData bin20 = MetricData.newBuilder()
                .setType(MetricType.IGAUGE)
                .setLabels(Labels.of("bin", "20", "project", "solomon", "sensor", "q3"))
                .addLong(now + 5_000,  2)
                .addLong(now + 10_000, 0)
                .addLong(now + 15_000, 1)
                .build();

        MetricData bin30 = MetricData.newBuilder()
                .setType(MetricType.IGAUGE)
                .setLabels(Labels.of("bin", "30", "project", "solomon", "sensor", "q3"))
                .addLong(now + 5_000,  1)
                .addLong(now + 10_000, 0)
                .addLong(now + 15_000, 5)
                .build();

        MetricData binInf = MetricData.newBuilder()
                .setType(MetricType.IGAUGE)
                .setLabels(Labels.of("bin", "inf", "project", "solomon", "sensor", "q3"))
                .addLong(now + 5_000,  0)
                .addLong(now + 10_000, 0)
                .addLong(now + 15_000, 8)
                .build();

        assertThat(result, Matchers.iterableWithSize(4));
        assertThat(result, hasItems(bin10, bin20, bin30, binInf));
    }

    private HistogramSnapshot hist(double[] bounds, long[] buckets) {
        return new ExplicitHistogramSnapshot(bounds, buckets);
    }

    private ByteBuf encode(long now, MetricRegistry registry) {
        ByteBufOutputStream out = new ByteBufOutputStream(UnpooledByteBufAllocator.DEFAULT.heapBuffer());
        try (MetricEncoder encoder = new MetricSpackEncoder(TimePrecision.SECONDS, CompressionAlg.NONE, out)) {
            registry.accept(now, encoder);
        } catch (Exception e) {
            throw new IllegalStateException("cannot encode metric", e);
        }

        return out.buffer();
    }

    private List<MetricData> decode(ByteBuf source) {
        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        MetricDataConsumer consumer = new MetricDataConsumer();
        TreeParser parser = TreeParserSpack.instance();
        parser.parseAndProcess(
                Labels.of("project", "solomon"),
                source,
                consumer,
                errorListener, TreeParser.FormatListenerIgnore.I, false);

        Assert.assertFalse(errorListener.anyErrors());
        return consumer.getMetrics();
    }
}
