package ru.yandex.solomon.metrics.parser.spack;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URL;
import java.time.Instant;
import java.util.List;

import com.google.common.io.Resources;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.MetricSpackEncoder;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.metrics.parser.TreeParser.ErrorListener;


/**
 * @author Sergey Polovko
 */
public class TreeParserSpackTest {

    private static final long COMMON_TS_MILLIS = Instant.parse("2017-08-27T12:34:56Z").toEpochMilli();

    private static final TimeSeries timeSeries = TimeSeries.newDouble(2)
            .addDouble(1503923187000L, 20.0)
            .addDouble(1503923531000L, -10.0);

    private static final MetricData[] EXPECTED_METRICS = {
            MetricData.newBuilder()
                    .setType(MetricType.DGAUGE)
                    .setLabels(Labels.of("project", "solomon", "sensor", "Memory"))
                    .addDouble(COMMON_TS_MILLIS, 10.0)
                    .build(),

            MetricData.newBuilder()
                    .setType(MetricType.COUNTER)
                    .setLabels(Labels.of("project", "solomon", "sensor", "UserTime"))
                    .addLong(COMMON_TS_MILLIS, 1)
                    .build(),

            MetricData.newBuilder()
                    .setType(MetricType.COUNTER)
                    .setLabels(Labels.of("project", "solomon", "sensor", "SystemTime"))
                    .addLong(COMMON_TS_MILLIS, -1)
                    .build(),

            MetricData.newBuilder()
                    .setType(MetricType.DGAUGE)
                    .setLabels(Labels.of("project", "solomon", "sensor", "QueueSize", "export", "Oxygen"))
                    .addDouble("2017-11-05T12:34:56.000Z", 3.14159)
                    .build(),

            MetricData.newBuilder()
                    .setType(MetricType.DGAUGE)
                    .setLabels(Labels.of("project", "solomon", "sensor", "Writes"))
                    .setTimeSeries(timeSeries)
                    .build(),
    };

    private void testParsing(String resourceName) {
        ByteBuf testData = loadResource(resourceName);
        ErrorListenerCounter errorListener = new ErrorListenerCounter();

        MetricDataConsumer consumer = new MetricDataConsumer();
        TreeParser parser = TreeParserSpack.instance();
        parser.parseAndProcess(
            Labels.of("project", "solomon"),
            testData,
            consumer,
            errorListener, TreeParser.FormatListenerIgnore.I, false);

        Assert.assertArrayEquals(EXPECTED_METRICS, consumer.getMetrics().toArray());
        Assert.assertFalse(errorListener.anyErrors());
    }

    @Test
    public void parseNotCompressed() {
        testParsing("test.sp");
    }

    @Test
    public void parseLz4Compressed() {
        testParsing("test.sp.lz4");
    }

    @Test
    public void parseZlibCompressed() {
        testParsing("test.sp.zlib");
    }

    @Test
    public void parseZstdCompressed() {
        testParsing("test.sp.zstd");
    }

    @Test
    public void tooManyLabels() {
        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        encodeMetricWithNLabels(buf, 14);

        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        List<MetricData> metrics = parse(buf.toByteArray(), errorListener);
        Assert.assertTrue(metrics.isEmpty());
        Assert.assertTrue(errorListener.anyErrors());
        Assert.assertEquals(1, errorListener.invalidMetrics.get(TreeParser.InvalidMetricReason.TOO_MANY_LABELS));
    }

    @Test
    public void notTooManyLabels() {
        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        encodeMetricWithNLabels(buf, 13);

        ErrorListenerCounter errorListener = new ErrorListenerCounter();
        List<MetricData> metrics = parse(buf.toByteArray(), errorListener);
        Assert.assertFalse(metrics.isEmpty());
        Assert.assertFalse(errorListener.anyErrors());

        MetricData metric = metrics.get(0);
        Assert.assertEquals(MetricType.DGAUGE, metric.getType());
        Assert.assertEquals(
            Labels.builder()
                .add("a0", "0")
                .add("a1", "1")
                .add("a2", "2")
                .add("a3", "3")
                .add("a4", "4")
                .add("a5", "5")
                .add("a6", "6")
                .add("a7", "7")
                .add("a8", "8")
                .add("a9", "9")
                .add("a10", "10")
                .add("a11", "11")
                .add("a12", "12")
                .build(),
            metric.getLabels());
        Assert.assertEquals(TimeSeries.newDouble(0, 3.14), metric.getTimeSeries());
    }

    private List<MetricData> parse(byte[] buf, ErrorListener errorListener) {
        MetricDataConsumer consumer = new MetricDataConsumer();
        TreeParserSpack parser = new TreeParserSpack(Labels.allocator, "");
        parser.parseAndProcess(
            Labels.of(),
            Unpooled.wrappedBuffer(buf),
            consumer,
            errorListener,
            TreeParser.FormatListenerIgnore.I,
            false);

        return consumer.getMetrics();
    }

    private void encodeMetricWithNLabels(ByteArrayOutputStream buf, int labelsCount) {
        try (var e = new MetricSpackEncoder(TimePrecision.SECONDS, CompressionAlg.NONE, buf)) {
            e.onStreamBegin(-1);
            {
                e.onMetricBegin(MetricType.DGAUGE);
                {
                    e.onLabelsBegin(labelsCount);
                    for (int i = 0; i < labelsCount; i++) {
                        e.onLabel("a" + i, Integer.toString(i));
                    }
                    e.onLabelsEnd();
                }
                e.onDouble(0, 3.14);
                e.onMetricEnd();
            }
            e.onStreamEnd();
        }
    }

    private static ByteBuf loadResource(String name) {
        try {
            URL resource = TreeParserSpackTest.class.getResource(name);
            return Unpooled.wrappedBuffer(Resources.toByteArray(resource));
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }
}
