package ru.yandex.solomon.metrics.parser.spack;

import java.io.IOException;
import java.net.URL;
import java.util.Collections;
import java.util.List;

import com.google.common.io.Resources;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.SpackException;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.MetricData;
import ru.yandex.solomon.metrics.parser.MetricDataConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;


/**
 * @author Oleg Baryshnikov
 */
public class TreeParserSpackWithMetricNameTest {

    @Test
    public void testMetricNameInOldFormat() {
        var expectedData = MetricData.newBuilder()
            .setType(MetricType.DGAUGE)
            .setLabels(Labels.of("project", "solomon", "sensor", "Memory"))
            .addDouble(0, 42)
            .build();

        List<MetricData> actualDatas = loadAndParse("test.sensor_name_in_old_format.sp", "sensor", false);

        Assert.assertEquals(Collections.singletonList(expectedData), actualDatas);
    }

    @Test
    public void testMetricNameWithoutLabels() {
        var expectedData = MetricData.newBuilder()
            .setType(MetricType.DGAUGE)
            .setLabels(Labels.of("project", "solomon", "sensor", "Memory"))
            .addDouble(0, 42)
            .build();

        List<MetricData> actualDatas = loadAndParse("test.sensor_name_without_labels.sp", "sensor", false);

        Assert.assertEquals(Collections.singletonList(expectedData), actualDatas);
    }

    @Test
    public void testMetricNameAndLabels() {
        var expectedData = MetricData.newBuilder()
            .setType(MetricType.DGAUGE)
            .setLabels(Labels.of("project", "solomon", "sensor", "Memory", "memory", "heap"))
            .addDouble(0, 42)
            .build();

        List<MetricData> actualDatas = loadAndParse("test.sensor_name_and_labels.sp", "sensor", false);

        Assert.assertEquals(Collections.singletonList(expectedData), actualDatas);
    }

    @Test(expected = SpackException.class)
    public void testDuplicatedMetricName() {
        loadAndParse("test.duplicated_sensor_name.sp", "sensor", false);
    }

    @Test(expected = SpackException.class)
    public void testUnknownMetricName() {
        loadAndParse("test.unknown_sensor_name.sp", "sensor", false);
    }

    @Test(expected = SpackException.class)
    public void testUnexpectedMetricName() {
        loadAndParse("test.unexpected_sensor_name.sp", "", false);
    }

    @Test(expected = SpackException.class)
    public void testOnlyNewFormatWrites() {
        loadAndParse("test.sensor_name_in_old_format.sp", "sensor", true);
    }

    private List<MetricData> loadAndParse(String resourceName, String metricNameLabel, boolean onlyNewFormatWrites) {
        ByteBuf testData = loadResource(resourceName);
        ErrorListenerCounter errorListener = new ErrorListenerCounter();

        MetricDataConsumer consumer = new MetricDataConsumer();
        TreeParser parser = new TreeParserSpack(Labels.allocator, metricNameLabel);
        parser.parseAndProcess(
            Labels.of("project", "solomon"),
            testData,
            consumer,
            errorListener,
            TreeParser.FormatListenerIgnore.I,
            onlyNewFormatWrites
        );

        return consumer.getMetrics();
    }

    private static ByteBuf loadResource(String name) {
        try {
            URL resource = TreeParserSpackWithMetricNameTest.class.getResource(name);
            return Unpooled.wrappedBuffer(Resources.toByteArray(resource));
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }
}
