package ru.yandex.solomon.name.resolver.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverClientStub implements NameResolverClient {
    private static final Logger logger = LoggerFactory.getLogger(NameResolverClientStub.class);
    private final ConcurrentHashMap<String, Shard> shardById = new ConcurrentHashMap<>();
    private boolean failed;

    public void addResource(String shardId, Resource... resources) {
        addResource(shardId, Arrays.asList(resources));
    }

    public void addResource(String shardId, List<Resource> resources) {
        gerOrCreateShard(shardId).addResource(resources);
    }

    public List<Resource> getResource(String shardId, String... resourceIds) {
        return gerOrCreateShard(shardId).getResources(resourceIds);
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        if (failed) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("failed"));
        }
        return CompletableFuture.supplyAsync(() -> gerOrCreateShard(request.cloudId).find(request));
    }

    @Override
    public CompletableFuture<ResolveResponse> resolve(ResolveRequest request) {
        if (failed) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("failed"));
        }
        return CompletableFuture.supplyAsync(() -> gerOrCreateShard(request.cloudId).resolve(request));
    }

    @Override
    public CompletableFuture<Void> update(UpdateRequest request) {
        if (failed) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("failed"));
        }
        return CompletableFuture.runAsync(() -> gerOrCreateShard(request.cloudId).addResource(request.resources));
    }

    @Override
    public CompletableFuture<ShardsResponse> getShardIds() {
        if (failed) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("failed"));
        }
        return CompletableFuture.supplyAsync(() -> new ShardsResponse(shardById.keySet()));
    }

    private Shard gerOrCreateShard(String shardId) {
        return shardById.computeIfAbsent(shardId, Shard::new);
    }

    @Override
    public void close() {

    }

    public void setFailed(boolean failed) {
        this.failed = failed;
    }

    private static class Shard {
        private final String shardId;
        private final Map<String, Resource> resourceById = new ConcurrentHashMap<>();

        public Shard(String shardId) {
            this.shardId = shardId;
        }

        public FindResponse find(FindRequest request) {
            logger.info("find: {}", request);
            int remaining = request.limit == 0 ? Integer.MAX_VALUE : request.limit;
            var result = new ArrayList<Resource>(request.limit);
            for (var resource : resourceById.values()) {
                if (!ResourceLabels.match(resource, request.selectors)) {
                    continue;
                }

                if (request.filterDeleted && resource.deletedAt != 0) {
                    continue;
                }

                if (request.filterReplaced && resource.replaced) {
                    continue;
                }

                if (remaining == 0) {
                    return new FindResponse(result, true, "");
                }

                remaining--;
                result.add(new Resource(resource));
            }

            return new FindResponse(result, false, "");
        }

        public ResolveResponse resolve(ResolveRequest request) {
            logger.info("resolve: {}", request);
            var result = new ArrayList<Resource>(request.resourceIds.size());
            for (var resourceId : request.resourceIds) {
                var resource = resourceById.get(resourceId);
                if (resource != null) {
                    result.add(new Resource(resource));
                }
            }

            return new ResolveResponse(result);
        }

        public void addResource(List<Resource> resources) {
            for (var resource : resources) {
                resourceById.put(resource.resourceId, new Resource(resource));
            }
        }

        public List<Resource> getResources(String[] resourceIds) {
            List<Resource> result = new ArrayList<>();
            for (String resourceId : resourceIds) {
                result.add(resourceById.get(resourceId));
            }
            return result;
        }
    }
}
