package ru.yandex.solomon.name.resolver.client;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.base.Strings;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
public class Resource extends DefaultObject {
    public String cloudId;
    public String folderId;
    public String service;
    public String type;
    public String resourceId;
    public String name;
    public String responsible = "";
    public String environment = "";
    public Map<String, String> resourceComplexId = Map.of();
    public long updatedAt;
    public long deletedAt;
    public long reindexAt;
    public boolean replaced;
    public Severity severity = Severity.UNKNOWN;

    public Resource() {
    }

    public Resource(Resource copy) {
        this.cloudId = copy.cloudId;
        this.folderId = copy.folderId;
        this.service = copy.service;
        this.type = copy.type;
        this.resourceId = copy.resourceId;
        this.name = copy.name;
        this.updatedAt = copy.updatedAt;
        this.deletedAt = copy.deletedAt;
        this.replaced = copy.replaced;
        this.reindexAt = copy.reindexAt;
        this.responsible = copy.responsible;
        this.environment = copy.environment;
        this.severity = Nullables.orDefault(copy.severity, Severity.UNKNOWN);
        this.resourceComplexId = new HashMap<>(Nullables.orEmpty(copy.resourceComplexId));
    }

    public Resource setCloudId(String cloudId) {
        this.cloudId = cloudId;
        return this;
    }

    public Resource setFolderId(String folderId) {
        this.folderId = folderId;
        return this;
    }

    public Resource setService(String service) {
        this.service = service;
        return this;
    }

    public Resource setType(String type) {
        this.type = type;
        return this;
    }

    public Resource setResourceComplexId(Map<String, String> resourceComplexId) {
        this.resourceComplexId = resourceComplexId;
        return this;
    }

    public Resource setSeverity(Severity severity) {
        this.severity = severity;
        return this;
    }

    public Resource setResourceId(String resourceId) {
        this.resourceId = resourceId;
        return this;
    }

    public Resource setName(String name) {
        this.name = name;
        return this;
    }

    public Resource setUpdatedAt(long updatedAt) {
        this.updatedAt = updatedAt;
        return this;
    }

    public Resource setDeletedAt(long deletedAt) {
        this.deletedAt = deletedAt;
        return this;
    }

    public Resource setReindexAt(long reindexAt) {
        this.reindexAt = reindexAt;
        return this;
    }

    public Resource setReplaced(boolean replaced) {
        this.replaced = replaced;
        return this;
    }

    public Resource setEnvironment(String environment) {
        this.environment = environment;
        return this;
    }

    public Resource setResponsible(String responsible) {
        this.responsible = responsible;
        return this;
    }

    public boolean hasName() {
        return !Strings.isNullOrEmpty(name);
    }

    @JsonIgnore
    public String getNameOrId() {
        if (hasName()) {
            return name;
        }

        return resourceId;
    }

    @JsonIgnore
    public String getResourceId() {
        return resourceId;
    }

    @Override
    public String toString() {
        return "Resource{" +
                "cloudId='" + cloudId + '\'' +
                ", folderId='" + folderId + '\'' +
                ", service='" + service + '\'' +
                ", type='" + type + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", name='" + name + '\'' +
                ", updatedAt=" + (updatedAt == 0 ? "null" : Instant.ofEpochMilli(updatedAt).toString()) +
                ", deletedAt=" + (deletedAt == 0 ? "null" : Instant.ofEpochMilli(deletedAt).toString()) +
                ", reindexAt=" + (reindexAt == 0 ? "null" : Instant.ofEpochMilli(reindexAt).toString()) +
                ", replaced=" + replaced +
                ", resourceComplexId=" + resourceComplexId +
                ", severity=" + severity +
                ", responsible=" + responsible +
                ", environment=" + environment +
                '}';
    }

    public enum Severity {
        UNKNOWN,
        HIGHLY_CRITICAL,
        CRITICAL,
        @Deprecated
        NORMAL,
        NON_CRITICAL
    }
}
