package ru.yandex.solomon.name.resolver.client.grpc;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.Status.Code;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.cluster.discovery.ClusterDiscoveryImpl;
import ru.yandex.cluster.discovery.GrpcClusterDiscovery;
import ru.yandex.discovery.DiscoveryService;
import ru.yandex.grpc.utils.GrpcClientOptions;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.FindResponse;
import ru.yandex.solomon.name.resolver.client.NameResolverClient;
import ru.yandex.solomon.name.resolver.client.ResolveRequest;
import ru.yandex.solomon.name.resolver.client.ResolveResponse;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.ShardsResponse;
import ru.yandex.solomon.name.resolver.client.UpdateRequest;
import ru.yandex.solomon.name.resolver.protobuf.ResourceServiceGrpc;
import ru.yandex.solomon.name.resolver.protobuf.ServerStatusRequest;
import ru.yandex.solomon.util.actors.PingActorRunner;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcNameResolverClient implements NameResolverClient {
    private final Logger logger = LoggerFactory.getLogger(GrpcNameResolverClient.class);
    private final ClusterDiscovery<GrpcTransport> discovery;
    private final PingActorRunner serverStatusActor;
    private volatile AssignmentsState assignments = new AssignmentsState();
    @Nullable
    private volatile String leader;

    public GrpcNameResolverClient(List<String> addresses, GrpcClientOptions opts) {
        var timer = opts.getTimer().orElseThrow(() -> new IllegalStateException("Timer not configured"));
        var executor = opts.getRpcExecutor().orElseThrow(() -> new IllegalStateException("Executor not configured"));
        this.discovery = GrpcClusterDiscovery.of(opts, addresses, timer, executor, TimeUnit.HOURS.toMillis(1L));
        this.serverStatusActor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofSeconds(15))
                .timer(timer)
                .executor(executor)
                .onPing(this::serverStatus)
                .build();

        serverStatusActor.forcePing();
    }

    public GrpcNameResolverClient(DiscoveryService discovery, List<String> addresses, GrpcClientOptions opts) {
        var timer = opts.getTimer().orElseThrow(() -> new IllegalStateException("Timer not configured"));
        var executor = opts.getRpcExecutor().orElseThrow(() -> new IllegalStateException("Executor not configured"));
        this.discovery = new ClusterDiscoveryImpl<>(
                address -> new GrpcTransport(address, opts),
                addresses,
                discovery,
                timer,
                executor,
                TimeUnit.HOURS.toMillis(1L));
        this.serverStatusActor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofSeconds(15))
                .timer(timer)
                .executor(executor)
                .onPing(this::serverStatus)
                .build();

        serverStatusActor.forcePing();
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request) {
        var assignments = this.assignments;
        if (assignments.stateHash == 0) {
            return failedFuture(Status.UNAVAILABLE.withDescription("assignment state not initialized yet").asException());
        }

        var address = assignments.nodeByShardId.get(request.cloudId);
        if (address == null) {
            return completedFuture(new FindResponse(List.of(), false, ""));
        }

        return unaryCall(address, ResourceServiceGrpc.getFindMethod(), Proto.toProto(request), request.expiredAt)
                .thenApply(Proto::fromProto)
                .thenApply(resolveResponse -> {
                    for (Resource resource : resolveResponse.resources) {
                        resource.cloudId = request.cloudId;
                    }
                    return resolveResponse;
                })
                .whenComplete((response, e) -> {
                    if (e != null) {
                        var status = Status.fromThrowable(e);
                        if (status.getCode() == Code.NOT_FOUND) {
                            // shard absent on host, time to check assignments
                            serverStatusActor.forcePing();
                        }
                    } else if (logger.isDebugEnabled()) {
                        logger.debug("call find({}): {}", request, response);
                    }
                });
    }

    @Override
    public CompletableFuture<ResolveResponse> resolve(ResolveRequest request) {
        var assignments = this.assignments;
        if (assignments.stateHash == 0) {
            return failedFuture(Status.UNAVAILABLE.withDescription("assignment state not initialized yet").asException());
        }

        var address = assignments.nodeByShardId.get(request.cloudId);
        if (address == null) {
            return completedFuture(new ResolveResponse(List.of()));
        }

        return unaryCall(address, ResourceServiceGrpc.getResolveMethod(), Proto.toProto(request), request.expiredAt)
                .thenApply(Proto::fromProto)
                .thenApply(resolveResponse -> {
                    for (Resource resource : resolveResponse.resources) {
                        resource.cloudId = request.cloudId;
                    }
                    return resolveResponse;
                })
                .whenComplete((response, e) -> {
                    if (e != null) {
                        var status = Status.fromThrowable(e);
                        if (status.getCode() == Code.NOT_FOUND) {
                            // shard absent on host, time to check assignments
                            serverStatusActor.forcePing();
                        }
                    } else if (logger.isDebugEnabled()) {
                        logger.debug("call resolve({}): {}", request, response);
                    }
                });
    }

    @Override
    public CompletableFuture<Void> update(UpdateRequest request) {
        var assignments = this.assignments;
        if (assignments.stateHash == 0) {
            return failedFuture(Status.UNAVAILABLE.withDescription("assignment state not initialized yet").asException());
        }
        return unaryCall(leader, ResourceServiceGrpc.getResolveShardAndUpdateResourcesMethod(), Proto.toProto(request), 0)
                .whenComplete((unused, e) -> {
                    if (e != null) {
                        var status = Status.fromThrowable(e);
                        if (status.getCode() == Code.NOT_FOUND) {
                            // shard absent on host, time to check assignments
                            serverStatusActor.forcePing();
                        }
                    } else if (logger.isDebugEnabled()) {
                        logger.debug("call update({}): {}", request, unused);
                    }
                })
                .thenApply(updateResourcesResponse -> null);
    }

    @Override
    public CompletableFuture<ShardsResponse> getShardIds() {
        var assignments = this.assignments;
        if (assignments.stateHash == 0) {
            return failedFuture(Status.UNAVAILABLE.withDescription("assignment state not initialized yet").asException());
        }
        return safeCall(() -> CompletableFuture.completedFuture(new ShardsResponse(assignments.nodeByShardId.keySet())));
    }

    private <ReqT, RespT> CompletableFuture<RespT> unaryCall(String target, MethodDescriptor<ReqT, RespT> method, ReqT request, long expiredAt) {
        try {
            GrpcTransport transport = discovery.getTransportByNode(target);
            return transport.unaryCall(method, request, expiredAt);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public void close() {
        serverStatusActor.close();
    }

    private CompletableFuture<?> serverStatus(int attempt) {
        var leaderCopy = leader;
        var transport = attempt > 0 || Strings.isNullOrEmpty(leaderCopy)
                ? discovery.getTransport()
                : discovery.getTransportByNode(leaderCopy);

        return serverStatus(transport)
                .thenAccept(response -> {
                    if (assignments.stateHash != response.stateHash && response.stateHash != 0) {
                        assignments = new AssignmentsState(response);
                    }
                    leader = response.leader;
                });
    }

    private CompletableFuture<ServerStatusResponse> serverStatus(GrpcTransport node) {
        var req = ServerStatusRequest.newBuilder()
                .setStateHash(assignments.stateHash)
                .setExpiredAt(System.currentTimeMillis() + 30_000)
                .build();

        var doneFuture = new CompletableFuture<ServerStatusResponse>();
        node.serverStreamingCall(
                ResourceServiceGrpc.getServerStatusMethod(),
                req,
                new ServerStatusObserver(doneFuture),
                req.getExpiredAt());
        return doneFuture;
    }

    private static class AssignmentsState {
        private final Map<String, String> nodeByShardId;
        private final long stateHash;

        public AssignmentsState() {
            nodeByShardId = Map.of();
            stateHash = 0;
        }

        public AssignmentsState(ServerStatusResponse response) {
            nodeByShardId = response.shardsByNode.entries()
                    .stream()
                    .collect(Collectors.toMap(Entry::getValue, Entry::getKey));
            stateHash = response.stateHash;
        }
    }
}
