package ru.yandex.solomon.name.resolver.client.grpc;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import io.grpc.Status;

import ru.yandex.solomon.common.StringPool.Compression;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.FindResponse;
import ru.yandex.solomon.name.resolver.client.ResolveRequest;
import ru.yandex.solomon.name.resolver.client.ResolveResponse;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.client.UpdateRequest;
import ru.yandex.solomon.name.resolver.protobuf.ResourcePooled;
import ru.yandex.solomon.name.resolver.protobuf.ResourceSeverity;
import ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse.Assignments;
import ru.yandex.solomon.name.resolver.protobuf.UpdateResourcesRequest;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.protobuf.StringPool;

/**
 * @author Vladimir Gordiychuk
 */
public class Proto {
    public static ru.yandex.solomon.name.resolver.protobuf.FindRequest toProto(FindRequest request) {
        return ru.yandex.solomon.name.resolver.protobuf.FindRequest.newBuilder()
                .setCloudId(request.cloudId)
                .setSelectors(SelectorsFormat.format(request.selectors))
                .setFilterDeleted(request.filterDeleted)
                .setFilterReplaced(request.filterReplaced)
                .setLimit(request.limit)
                .setExpiredAt(request.expiredAt)
                .setNextPageToken(request.pageToken)
                .build();
    }

    public static FindRequest fromProto(ru.yandex.solomon.name.resolver.protobuf.FindRequest request) {
        try {
            return FindRequest.newBuilder()
                    .cloudId(request.getCloudId())
                    .selectors(Selectors.parse(request.getSelectors()))
                    .filterDeleted(request.getFilterDeleted())
                    .filterReplaced(request.getFilterReplaced())
                    .limit(request.getLimit())
                    .expiredAt(request.getExpiredAt())
                    .pageToken(request.getNextPageToken())
                    .build();
        } catch (Throwable e) {
            throw Status.INVALID_ARGUMENT.withCause(e).asRuntimeException();
        }
    }

    public static ru.yandex.solomon.name.resolver.protobuf.ResolveRequest toProto(ResolveRequest request) {
        return ru.yandex.solomon.name.resolver.protobuf.ResolveRequest.newBuilder()
                .setCloudId(request.cloudId)
                .addAllResourceIds(request.resourceIds)
                .setExpiredAt(request.expiredAt)
                .build();
    }

    public static ResolveRequest fromProto(ru.yandex.solomon.name.resolver.protobuf.ResolveRequest request) {
        try {
            return ResolveRequest.newBuilder()
                    .cloudId(request.getCloudId())
                    .resourceIds(request.getResourceIdsList())
                    .expiredAt(request.getExpiredAt())
                    .build();
        } catch (Throwable e) {
            throw Status.INVALID_ARGUMENT.withCause(e).asRuntimeException();
        }
    }

    public static ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse toProto(ServerStatusResponse snapshot) {
        return ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse.newBuilder()
                .setStateHash(snapshot.stateHash)
                .setLeader(snapshot.leader)
                .addAllAssignments(snapshot.shardsByNode.asMap().entrySet()
                        .stream()
                        .map(entry -> Assignments.newBuilder()
                                .setAddress(entry.getKey())
                                .addAllShardId(entry.getValue())
                                .build())
                        .collect(Collectors.toList()))
                .build();
    }

    public static ServerStatusResponse fromProto(ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse proto) {
        try {
            var result = new ServerStatusResponse(proto.getStateHash(), proto.getLeader());
            for (var assignments : proto.getAssignmentsList()) {
                result.shardsByNode.putAll(assignments.getAddress(), assignments.getShardIdList());
            }
            return result;
        } catch (Throwable e) {
            throw Status.INVALID_ARGUMENT.withCause(e).asRuntimeException();
        }
    }

    public static ru.yandex.solomon.name.resolver.protobuf.FindResponse toProto(FindResponse response) {
        var stringPool = StringPool.newBuilder();
        var result = ru.yandex.solomon.name.resolver.protobuf.FindResponse.newBuilder();
        for (var resource : response.resources) {
            result.addResources(toProto(stringPool, resource));
        }
        result.setStringPool(stringPool.buildProto(Compression.LZ4));
        result.setTruncated(response.truncated);
        result.setNextPageToken(response.nextPageToken);
        return result.build();
    }

    public static FindResponse fromProto(ru.yandex.solomon.name.resolver.protobuf.FindResponse proto) {
        var stringPool = StringPool.fromProto(proto.getStringPool());
        var result = new ArrayList<Resource>(proto.getResourcesCount());
        for (var protoResource : proto.getResourcesList()) {
            result.add(fromProto(stringPool, protoResource));
        }
        return new FindResponse(result, proto.getTruncated(), proto.getNextPageToken());
    }

    public static ru.yandex.solomon.name.resolver.protobuf.ResolveResponse toProto(ResolveResponse response) {
        var stringPool = StringPool.newBuilder();
        var result = ru.yandex.solomon.name.resolver.protobuf.ResolveResponse.newBuilder();
        for (var resource : response.resources) {
            result.addResources(toProto(stringPool, resource));
        }
        result.setStringPool(stringPool.buildProto(Compression.LZ4));
        return result.build();
    }

    public static UpdateRequest fromProto(UpdateResourcesRequest response) {
        String cloudId = response.getCloudId();
        String service = null;
        var resources = new ArrayList<Resource>(response.getResourcesCount());
        for (var proto : response.getResourcesList()) {
            final Resource resource = fromProto(cloudId, proto);
            if (service == null) {
                service = resource.service;
            } else if (!service.equals(resource.service) && response.getRemoveOtherOfService()) {
                throw new IllegalArgumentException("Cant delete multiple services resources");
            }
            resources.add(resource);
        }
        return new UpdateRequest(cloudId, resources, response.getRemoveOtherOfService(), response.getServiceProviderId());
    }

    public static ResolveResponse fromProto(ru.yandex.solomon.name.resolver.protobuf.ResolveResponse proto) {
        var stringPool = StringPool.fromProto(proto.getStringPool());
        var result = new ArrayList<Resource>(proto.getResourcesCount());
        for (var protoResource : proto.getResourcesList()) {
            result.add(fromProto(stringPool, protoResource));
        }
        return new ResolveResponse(result);
    }

    public static ResourcePooled toProto(StringPool.Builder stringPool, Resource resource) {
        return ResourcePooled.newBuilder()
                .setFolderIdIdx(stringPool.put(resource.folderId))
                .setServiceIdx(stringPool.put(resource.service))
                .setTypeIdx(stringPool.put(resource.type))
                .setResourceId(resource.resourceId)
                .setNameIdx(stringPool.put(resource.name))
                .setUpdatedAt(resource.updatedAt)
                .setDeletedAt(resource.deletedAt)
                .setReplaced(resource.replaced)
                .putAllResourceComplexId(resource.resourceComplexId)
                .setSeverity(severity(resource.severity))
                .setEnvironmentIdx(stringPool.put(resource.environment))
                .setResponsibleIdx(stringPool.put(resource.responsible))
                .build();
    }

    public static Resource fromProto(StringPool stringPool, ResourcePooled proto) {
        return new Resource()
                .setCloudId("")
                .setFolderId(stringPool.get(proto.getFolderIdIdx()))
                .setService(stringPool.get(proto.getServiceIdx()))
                .setType(stringPool.get(proto.getTypeIdx()))
                .setResourceId(proto.getResourceId())
                .setName(stringPool.get(proto.getNameIdx()))
                .setUpdatedAt(proto.getUpdatedAt())
                .setResourceComplexId(proto.getResourceComplexIdMap())
                .setSeverity(severity(proto.getSeverity()))
                .setResponsible(stringPool.get(proto.getResponsibleIdx()))
                .setEnvironment(stringPool.get(proto.getEnvironmentIdx()))
                .setDeletedAt(proto.getDeletedAt())
                .setReplaced(proto.getReplaced());
    }

    public static ru.yandex.solomon.name.resolver.protobuf.Resource toProto(Resource resource) {
        return ru.yandex.solomon.name.resolver.protobuf.Resource.newBuilder()
                .setFolderId(resource.folderId)
                .setService(resource.service)
                .setType(resource.type)
                .setResourceId(resource.resourceId)
                .setName(Nullables.orEmpty(resource.name))
                .setUpdatedAt(resource.updatedAt)
                .setDeletedAt(resource.deletedAt)
                .setReindexAt(resource.reindexAt)
                .putAllResourceComplexId(resource.resourceComplexId)
                .setSeverity(severity(resource.severity))
                .setEnvironment(resource.environment)
                .setResponsible(resource.responsible)
                .build();
    }

    public static Resource fromProto(String cloudId, ru.yandex.solomon.name.resolver.protobuf.Resource proto) {
        var result = new Resource();
        result.cloudId = cloudId;
        result.folderId = proto.getFolderId();
        result.service = proto.getService();
        result.type = proto.getType();
        result.resourceId = proto.getResourceId();
        result.name = proto.getName();
        result.updatedAt = proto.getUpdatedAt();
        result.deletedAt = proto.getDeletedAt();
        result.reindexAt = proto.getReindexAt();
        result.resourceComplexId = proto.getResourceComplexIdMap();
        result.severity = severity(proto.getSeverity());
        result.responsible = proto.getResponsible();
        result.environment = proto.getEnvironment();
        return result;
    }

    public static List<ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse> split(ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse proto, int limitBytes) {
        if (proto.getSerializedSize() <= limitBytes) {
            return List.of(proto);
        }

        List<ru.yandex.solomon.name.resolver.protobuf.ServerStatusResponse> result = new ArrayList<>();
        var chunk = proto.toBuilder().clearAssignments();
        for (var assignments : proto.getAssignmentsList()) {
            for (var assignmentChunk : split(assignments, limitBytes)) {
                chunk.addAssignments(assignmentChunk);
                result.add(chunk.build());
                chunk.clearAssignments();
            }
        }

        return result;
    }

    public static List<Assignments> split(Assignments proto, int limitBytes) {
        if (proto.getSerializedSize() <= limitBytes) {
            return List.of(proto);
        }

        var result = new ArrayList<Assignments>();
        var chunk = proto.toBuilder().clearShardId();
        int bytes = 0;
        for (int index = 0; index < proto.getShardIdCount(); index++) {
            var shardId = proto.getShardIdBytes(index);
            bytes += shardId.size();
            chunk.addShardIdBytes(shardId);
            if (bytes >= limitBytes) {
                result.add(chunk.build());
                chunk.clearShardId();
                bytes = 0;
            }
        }

        if (bytes != 0) {
            result.add(chunk.build());
        }

        return result;
    }

    public static UpdateResourcesRequest toProto(UpdateRequest request) {
        String cloudId = request.cloudId;
        var resources = new ArrayList<ru.yandex.solomon.name.resolver.protobuf.Resource>(request.resources.size());
        for (var resource : request.resources) {
            resources.add(toProto(resource));
        }
        return UpdateResourcesRequest.newBuilder()
                .addAllResources(resources)
                .setCloudId(cloudId)
                .setRemoveOtherOfService(request.removeOther)
                .setServiceProviderId(request.serviceProviderId)
                .build();
    }

    private static ResourceSeverity severity(Resource.Severity severity) {
        switch (severity) {
            case HIGHLY_CRITICAL -> {
                return ResourceSeverity.RESOURCE_SEVERITY_HIGHLY_CRITICAL;
            }
            case CRITICAL -> {
                return ResourceSeverity.RESOURCE_SEVERITY_CRITICAL;
            }
            case NON_CRITICAL -> {
                return ResourceSeverity.RESOURCE_SEVERITY_NON_CRITICAL;
            }
            case NORMAL -> {
                return ResourceSeverity.RESOURCE_SEVERITY_NON_CRITICAL;
            }
            default -> {
                return ResourceSeverity.RESOURCE_SEVERITY_UNSPECIFIED;
            }
        }
    }

    private static Resource.Severity severity(ResourceSeverity severity) {
        switch (severity) {
            case RESOURCE_SEVERITY_HIGHLY_CRITICAL -> {
                return Resource.Severity.HIGHLY_CRITICAL;
            }
            case RESOURCE_SEVERITY_CRITICAL -> {
                return Resource.Severity.CRITICAL;
            }
            case RESOURCE_SEVERITY_NON_CRITICAL -> {
                return Resource.Severity.NON_CRITICAL;
            }
            default -> {
                return Resource.Severity.UNKNOWN;
            }
        }
    }
}
