package ru.yandex.solomon.name.resolver.client;

import java.time.Clock;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.ut.ManualClock;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class CrossDcNameResolverClientTest {

    private Unit alice;
    private Unit bob;
    private NameResolverClient client;
    private Response ok;
    private Response fail;
    private CrossDcNameResolverClient crossDcNameResolverClient;
    private Clock clock;

    @Before
    public void setUp() throws Exception {
        alice = new Unit("alice");
        bob = new Unit("bob");
        clock = new ManualClock();
        crossDcNameResolverClient = new CrossDcNameResolverClient(clock, ImmutableMap.of(
                alice.name, alice.resolver,
                bob.name, bob.resolver));
        client = crossDcNameResolverClient;
        ok = new Response(true);
        fail = new Response(false);
    }

    @Test
    public void successFind_merge() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource().setResourceId("2"));

        FindResponse response = client.find(FindRequest.newBuilder()
                .cloudId("solomon")
                .build())
                .join();

        assertEquals(response.resources, List.of(staticResource(), staticResource().setResourceId("2")));
    }

    @Test
    public void successFind_one() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        bob.failed(true);

        FindResponse response = client.find(FindRequest.newBuilder()
                .cloudId("solomon")
                .build())
                .join();

        assertEquals(response.resources, List.of(staticResource()));
    }

    @Test(expected = IllegalArgumentException.class)
    public void failedFind() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        alice.failed(true);
        bob.failed(true);
        try {
            client.find(FindRequest.newBuilder()
                    .cloudId("solomon")
                    .build())
                    .join();
        } catch (CompletionException e) {
            Throwables.propagate(e.getCause());
        }
    }

    @Test
    public void successResolve_merge() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource().setResourceId("2"));

        var response = client.resolve(ResolveRequest.newBuilder()
                .cloudId("solomon")
                .resourceIds(List.of("2", "resourceId"))
                .build())
                .join();

        assertEquals(response.resources, List.of(staticResource(), staticResource().setResourceId("2")));
    }

    @Test
    public void successResolve_one() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource().setResourceId("2"));
        bob.failed(true);

        var response = client.resolve(ResolveRequest.newBuilder()
                .cloudId("solomon")
                .resourceIds(List.of("2", "resourceId"))
                .build())
                .join();

        assertEquals(response.resources, List.of(staticResource()));
    }

    @Test(expected = IllegalArgumentException.class)
    public void failedResolve() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        alice.failed(true);
        bob.failed(true);
        try {
            client.resolve(ResolveRequest.newBuilder()
                    .cloudId("solomon")
                    .resourceIds(List.of("2", "resourceId"))
                    .build())
                    .join();
        } catch (CompletionException e) {
            Throwables.propagate(e.getCause());
        }
    }

    @Test
    public void successUpdate_merge() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource().setResourceId("2"));

        client.update(new UpdateRequest("solomon", List.of(
                staticResource().setFolderId("folder2"),
                staticResource().setResourceId("2").setFolderId("folder23")
        ))).join();


        assertEquals(alice.resolver.getResource("solomon", "resourceId"), List.of(staticResource().setFolderId("folder2")));
        assertEquals(bob.resolver.getResource("solomon", "resourceId", "2"), List.of(staticResource().setFolderId("folder2"), staticResource().setFolderId("folder23").setResourceId("2")));
    }

    @Test
    public void successUpdate_one() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource().setResourceId("2"));
        bob.failed(true);

        client.update(new UpdateRequest("solomon", List.of(
                staticResource().setFolderId("folder2")
        ))).join();


        assertEquals(alice.resolver.getResource("solomon", "resourceId"), List.of(staticResource().setFolderId("folder2")));
        assertEquals(bob.resolver.getResource("solomon", "resourceId", "2"), List.of(staticResource(), staticResource().setResourceId("2")));
    }

    @Test(expected = IllegalArgumentException.class)
    public void failedUpdate() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        alice.failed(true);
        bob.failed(true);
        try {
            client.update(new UpdateRequest("solomon", List.of(
                    staticResource().setFolderId("folder2")
            ))).join();
        } catch (CompletionException e) {
            Throwables.propagate(e.getCause());
        }
    }

    @Test
    public void successGetShards_merge() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon2", staticResource());
        bob.resolver.addResource("solomon", staticResource().setResourceId("2"));

        final ShardsResponse join = client.getShardIds().join();
        assertEquals(Set.of("solomon", "solomon2"), join.ids());
    }

    @Test
    public void successGetShards_one() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon2", staticResource());
        bob.resolver.addResource("solomon", staticResource().setResourceId("2"));
        bob.failed(true);

        final ShardsResponse join = client.getShardIds().join();
        assertEquals(Set.of("solomon"), join.ids());
    }

    @Test(expected = IllegalArgumentException.class)
    public void failedGetShards() {
        alice.resolver.addResource("solomon", staticResource());
        bob.resolver.addResource("solomon", staticResource());
        alice.failed(true);
        bob.failed(true);
        try {
            client.getShardIds().join();
        } catch (CompletionException e) {
            Throwables.propagate(e.getCause());
        }
    }

    @Test
    public void anyOrAllOf1() {
        // OK
        {
            CompletableFuture<List<Response>> responsesFuture = crossDcNameResolverClient
                    .anyOrAllOf(List.of(completedFuture(ok)), clock.millis() + 10_000);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            Assert.assertEquals(List.of(ok), responsesFuture.join());
        }

        // FAIL
        {
            CompletableFuture<List<Response>> responsesFuture = crossDcNameResolverClient
                    .anyOrAllOf(List.of(completedFuture(fail)), clock.millis() + 10_000);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            Assert.assertEquals(List.of(fail), responsesFuture.join());
        }
    }

    @Test
    public void anyOrAllOf2() {
        long normal = 200;
        long tooLong = 600;
        long max = 2000;

        // OK, FAIL
        {
            CompletableFuture<List<Response>> responsesFuture = crossDcNameResolverClient.anyOrAllOf(List.of(
                    completedFuture(ok),
                    completedFuture(fail)),
                    clock.millis() + max);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            Assert.assertEquals(List.of(ok, fail), responsesFuture.join());
        }

        // OK, FAIL (delayed < maxTime)
        {
            List<Response> responses = crossDcNameResolverClient.anyOrAllOf(List.of(
                    delayedResponse(fail, normal),
                    completedFuture(ok)),
                    clock.millis() + tooLong).join();
            Assert.assertEquals(List.of(fail, ok), responses);
        }


        // OK, FAIL (delayed > maxTime)
        {
            List<Response> responses = crossDcNameResolverClient.anyOrAllOf(List.of(
                    delayedResponse(fail, tooLong),
                    completedFuture(ok)),
                    clock.millis() + normal).join();
            Assert.assertEquals(List.of(ok), responses);
        }

        // OK (delayed > maxTime), FAIL
        {
            List<Response> responses = crossDcNameResolverClient.anyOrAllOf(List.of(
                    delayedResponse(ok, tooLong),
                    completedFuture(fail)),
                    clock.millis() + normal).join();
            // anyway we await OK response
            Assert.assertEquals(List.of(fail), responses);
        }

        // OK (delayed > maxTime), FAIL
        {
            List<Response> responses = crossDcNameResolverClient.anyOrAllOf(List.of(
                    delayedResponse(ok, tooLong),
                    completedFuture(fail)),
                    0).join();
            // anyway we await OK response
            Assert.assertEquals(List.of(ok, fail), responses);
        }
    }

    @Test
    public void anyOrAllOf3() {
        long fast = 100;
        long normal = 200;
        long tooLong = 1000;

        {
            List<Response> responses = crossDcNameResolverClient.anyOrAllOf(List.of(
                    completedFuture(ok.from(1)),
                    delayedResponse(ok.from(2), tooLong),
                    delayedResponse(fail.from(3), fast)
            ), clock.millis() + normal).join();

            Assert.assertEquals(List.of(ok.from(1), fail.from(3)), responses);
        }

        {
            List<Response> responses = crossDcNameResolverClient.anyOrAllOf(List.of(
                    completedFuture(ok.from(1)),
                    delayedResponse(fail.from(2), tooLong),
                    delayedResponse(fail.from(3), fast)
            ), clock.millis() + normal).join();

            Assert.assertEquals(List.of(ok.from(1), fail.from(3)), responses);
        }

        {
            List<Response> responses = crossDcNameResolverClient.anyOrAllOf(List.of(
                    delayedResponse(ok.from(1), fast),
                    delayedResponse(fail.from(2), fast),
                    delayedResponse(fail.from(3), tooLong)
            ), clock.millis() + normal).join();

            Assert.assertEquals(List.of(ok.from(1), fail.from(2)), responses);
        }
    }

    private CompletableFuture<Response> delayedResponse(Response r, long delayMillis) {
        CompletableFuture<Response> f = new CompletableFuture<>();
        CompletableFuture.delayedExecutor(delayMillis, TimeUnit.MILLISECONDS)
                .execute(() -> f.complete(r));
        return f;
    }

    private static final class Response {
        private final boolean ok;
        private final int id;

        private Response(boolean ok) {
            this(ok, 0);
        }

        private Response(boolean ok, int id) {
            this.ok = ok;
            this.id = id;
        }

        private Response from(int id) {
            return new Response(ok, id);
        }

        public boolean isOk() {
            return ok;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Response response = (Response) o;
            return ok == response.ok &&
                    id == response.id;
        }

        @Override
        public int hashCode() {
            return Objects.hash(ok, id);
        }

        @Override
        public String toString() {
            return "Response{ok=" + ok + ", id=" + id + '}';
        }
    }

    private static class Unit {
        private final String name;
        private final NameResolverClientStub resolver;

        Unit(String name) {
            this.name = name;
            this.resolver = new NameResolverClientStub();
        }

        public void failed(boolean b) {
            resolver.setFailed(b);
        }
    }
}
