package ru.yandex.solomon.name.resolver.client;

import java.util.List;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.labels.query.Selectors;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class NameResolverClientStubTest {

    private NameResolverClientStub client;

    @Before
    public void setUp() throws Exception {
        client = new NameResolverClientStub();
    }

    @Test
    public void emptyFind() {
        var result = client.find(FindRequest.newBuilder()
                .cloudId("shard")
                .build())
                .join();

        assertEquals(new FindResponse(List.of(), false, ""), result);
    }

    @Test
    public void findMatcherResource() {
        var resource = staticResource().setResourceId("myId").setName("myName");
        client.addResource("myShard", resource);
        var result = client.find(FindRequest.newBuilder()
                .cloudId("myShard")
                .selectors(Selectors.parse("resource=myName"))
                .build())
                .join();

        assertEquals(new FindResponse(List.of(resource), false, ""), result);
    }

    @Test
    public void filterReplaced() {
        var resourceReplaced = staticResource().setResourceId("myId").setName("myName").setReplaced(true);
        var resource = staticResource().setResourceId("myId2").setName("myName");
        client.addResource("myShard", resource, resourceReplaced);
        var result = client.find(FindRequest.newBuilder()
                .cloudId("myShard")
                .selectors(Selectors.parse("resource=myName"))
                .filterReplaced(true)
                .build())
                .join();

        assertEquals(new FindResponse(List.of(resource), false, ""), result);
    }

    @Test
    public void findNoMatched() {
        var resource = staticResource().setResourceId("myId").setName("myName");
        client.addResource("myShard", resource);
        var result = client.find(FindRequest.newBuilder()
                .cloudId("myShard")
                .selectors(Selectors.parse("resource=myNotExistName"))
                .build())
                .join();

        assertEquals(new FindResponse(List.of(), false, ""), result);
    }

    @Test
    public void limitResult() {
        var resources = List.of(
                resource().setResourceId("a").setName("a"),
                resource().setResourceId("b").setName("b"),
                resource().setResourceId("c").setName("c"));

        client.addResource("alice", resources);
        {
            var result = client.find(FindRequest.newBuilder()
                    .cloudId("alice")
                    .selectors(Selectors.parse("resource=a|b|c"))
                    .limit(100)
                    .build())
                    .join();

            assertFalse(result.truncated);
            assertEquals(Set.copyOf(resources), Set.copyOf(result.resources));
        }

        {
            var result = client.find(FindRequest.newBuilder()
                    .cloudId("alice")
                    .selectors(Selectors.parse("resource=a|b|c"))
                    .limit(3)
                    .build())
                    .join();

            assertFalse(result.truncated);
            assertEquals(Set.copyOf(resources), Set.copyOf(result.resources));
        }

        {
            var result = client.find(FindRequest.newBuilder()
                    .cloudId("alice")
                    .selectors(Selectors.parse("resource=a|b|c"))
                    .limit(2)
                    .build())
                    .join();

            assertTrue(result.truncated);
            assertEquals(2, result.resources.size());
        }
    }

    private Resource resource() {
        return staticResource().setCloudId("");
    }
}
