package ru.yandex.solomon.name.resolver.client.grpc;

import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Test;

import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.FindResponse;
import ru.yandex.solomon.name.resolver.client.ResolveResponse;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.name.resolver.protobuf.UpdateResourcesRequest;
import ru.yandex.solomon.util.protobuf.StringPool;

import static org.hamcrest.Matchers.greaterThan;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class ProtoTest {

    @Test
    public void convertFindRequest() {
        var original = FindRequest.newBuilder()
                .cloudId("myCloud")
                .selectors(Selectors.parse("folder=solomon, service=compute, resource=alice"))
                .limit(42)
                .expiredAt(System.currentTimeMillis())
                .build();
        var proto = Proto.toProto(original);
        var converted = Proto.fromProto(proto);
        assertEquals(original, converted);
    }

    @Test
    public void convertFindRequestFilterReplaced() {
        var original = FindRequest.newBuilder()
                .cloudId("myCloud")
                .selectors(Selectors.parse("folder=solomon, service=compute, resource=a*"))
                .filterReplaced(true)
                .build();
        var proto = Proto.toProto(original);
        var converted = Proto.fromProto(proto);
        assertEquals(original, converted);
    }

    @Test
    public void convertServerStatusResponse() {
        var original = new ServerStatusResponse(123, "alice")
                .addShard("alice", "a")
                .addShard("alice", "b")
                .addShard("bob", "c");
        var proto = Proto.toProto(original);
        var restored = Proto.fromProto(proto);
        assertEquals(original, restored);
    }

    @Test
    public void skipSplitServerStatusResponse() {
        var original = new ServerStatusResponse(42, "test")
                .addShard("alice", "a")
                .addShard("alice", "b")
                .addShard("bob", "c");
        var proto = Proto.toProto(original);
        var split = Proto.split(proto, 10 << 20);
        assertEquals(List.of(proto), split);
    }

    @Test
    public void splitServerStatusResponse() {
        var original = new ServerStatusResponse(333, "leader");
        for (int hostIndex = 0; hostIndex < 10; hostIndex++) {
            String host = "solomon-" + hostIndex + ".yandex-team.ru";
            for (int shardIndex = 0; shardIndex < ThreadLocalRandom.current().nextInt(1000); shardIndex++) {
                original.addShard(host, hostIndex + "-shard-unique-id-" + shardIndex);
            }
        }
        var proto = Proto.toProto(original);
        var split = Proto.split(proto, 1024);
        assertThat(split.size(), greaterThan(1));

        var converted = split.stream()
                .map(Proto::fromProto)
                .reduce(ServerStatusResponse::combine)
                .orElse(null);
        assertEquals(original, converted);
    }

    @Test
    public void convertPooledResource() {
        var original = resource();
        var stringPool = StringPool.newBuilder();
        var proto = Proto.toProto(stringPool, original);
        var converted = Proto.fromProto(stringPool.build(), proto);
        assertEquals(original, converted);
    }

    @Test
    public void convertPooledResourceReplaced() {
        var original = resource().setReplaced(true);
        var stringPool = StringPool.newBuilder();
        var proto = Proto.toProto(stringPool, original);
        var converted = Proto.fromProto(stringPool.build(), proto);
        assertEquals(original, converted);
    }

    @Test
    public void convertFindResponse() {
        var original = new FindResponse(
                IntStream.range(0, 100)
                        .mapToObj(id -> resource().setResourceId("id-" + id).setName("vm-" + id))
                        .collect(Collectors.toList()),
                ThreadLocalRandom.current().nextBoolean(),"");
        var proto = Proto.toProto(original);
        var converted = Proto.fromProto(proto);
        assertEquals(original, converted);
    }

    @Test
    public void convertResolveResponse() {
        var original = new ResolveResponse(
                IntStream.range(0, 100)
                        .mapToObj(id -> resource().setResourceId("id-" + id).setName("vm-" + id))
                        .collect(Collectors.toList()));
        var proto = Proto.toProto(original);
        var converted = Proto.fromProto(proto);
        assertEquals(original, converted);
    }

    @Test
    public void convertResource() {
        var original = resource()
                .setDeletedAt(123)
                .setReindexAt(555);
        var proto = Proto.toProto(original);
        var converted = Proto.fromProto(original.cloudId, proto);
        assertEquals(original, converted);
    }

    @Test
    public void convertFromProto() {
        var req = UpdateResourcesRequest.newBuilder()
                .addAllResources(List.of())
                .setRemoveOtherOfService(true)
                .build();
        var result = Proto.fromProto(req);
        assertEquals(true, result.removeOther);

        req = UpdateResourcesRequest.newBuilder()
                .addAllResources(List.of(
                        Proto.toProto(resource()),
                        Proto.toProto(resource().setService("123"))
                ))
                .build();
        Proto.fromProto(req);

        try {
            Proto.fromProto(req.toBuilder().setRemoveOtherOfService(true).build());
            assertTrue(false);
        } catch (Exception e) {
            assertNotNull(e);
        }
    }

    private static Resource resource() {
        return staticResource().setCloudId("");
    }
}
