package ru.yandex.solomon.quotas.manager;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.core.db.dao.QuotasDao;
import ru.yandex.solomon.core.db.model.Quota;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
@LinkedOnRootPage("Quota Manager")
public class QuotaManager {
    private final QuotasDao quotasDao;

    private final Map<String, List<QuotaFetcher>> delegatesByNamespace = new HashMap<>();
    private final Map<String, String> namespaceByIndicator = new HashMap<>();

    public QuotaManager(QuotasDao quotasDao, List<QuotaFetcher> quotaFetchers) {
        this.quotasDao = quotasDao;
        for (var fetcher : quotaFetchers) {
            register(fetcher);
        }
    }

    @VisibleForTesting
    public QuotaManager(QuotasDao quotasDao, QuotaFetcher... quotaFetchers) {
        this(quotasDao, Arrays.stream(quotaFetchers).collect(Collectors.toList()));
    }

    @ManagerMethod
    public CompletableFuture<Void> updateLimit(
            @ManagerMethodArgument(name = "namespace") String namespace,
            @ManagerMethodArgument(name = "scopeType") String scopeType,
            @ManagerMethodArgument(name = "scopeId") String scopeId,
            @ManagerMethodArgument(name = "indicator") String indicator,
            @ManagerMethodArgument(name = "newLimit") long newLimit)
    {
        return updateLimit(Scope.of(namespace, scopeType, scopeId), indicator, newLimit, "manager-api", Instant.now());
    }

    @ManagerMethod
    public CompletableFuture<Void> updateDefaultLimit(
            @ManagerMethodArgument(name = "namespace") String namespace,
            @ManagerMethodArgument(name = "scopeType") String scopeType,
            @ManagerMethodArgument(name = "indicator") String indicator,
            @ManagerMethodArgument(name = "newLimit") long newLimit)
    {
        return updateLimit(Scope.defaultOf(namespace, scopeType), indicator, newLimit, "manager-api", Instant.now());
    }

    @ManagerMethod
    public CompletableFuture<Void> dropIndicator(
            @ManagerMethodArgument(name = "namespace") String namespace,
            @ManagerMethodArgument(name = "scopeType") String scopeType,
            @ManagerMethodArgument(name = "indicator") String indicator)
    {
        return deleteIndicator(namespace, scopeType, indicator);
    }

    @ManagerMethod
    public CompletableFuture<Void> resetLimitToDefault(
        @ManagerMethodArgument(name = "namespace") String namespace,
        @ManagerMethodArgument(name = "scopeType") String scopeType,
        @ManagerMethodArgument(name = "scopeId") String scopeId,
        @ManagerMethodArgument(name = "indicator") String indicator)
    {
        return resetLimitToDefault(Scope.of(namespace, scopeType, scopeId), indicator);
    }

    @ManagerMethod
    public CompletableFuture<List<QuotaLimit>> defaultLimits(
        @ManagerMethodArgument(name = "namespace") String namespace,
        @ManagerMethodArgument(name = "scopeType") String scopeType)
    {
        return getLimits(Scope.defaultOf(namespace, scopeType));
    }

    public CompletableFuture<List<QuotaValueWithLimit>> getUsageWithLimits(Scope scope) {
        if (scope.isDefaults()) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Cannot get quota usage for defaults"));
        }

        var futures = delegatesByNamespace.get(scope.getNamespace()).stream()
            .map(quotaFetcher -> quotaFetcher.fetch(scope))
            .collect(Collectors.toList());

        var valuesFuture = CompletableFutures.allOf(futures)
            .thenApply(allServiceQuotaValues -> allServiceQuotaValues.stream()
                .flatMap(List::stream)
                .collect(Collectors.toList()));

        var limitsFuture = getLimits(scope);

        return CompletableFutures.allOf2(valuesFuture, limitsFuture)
            .thenApply(this::mergeValuesWithLimits);
    }


    /**
     * Update limit for scope (Scope#of) or update defaults for scope (Scope#defaultOf)
     */
    public CompletableFuture<Void> updateLimit(Scope scope, String indicator, long newLimit, String updatedBy, Instant updatedAt) {
        return quotasDao.upsert(scope.getNamespace(), scope.getType(), scope.getIdentifier(), indicator, newLimit, updatedBy, updatedAt);
    }

    public CompletableFuture<List<QuotaLimit>> getDefaultLimits(String namespace, String scopeType) {
        return getLimits(Scope.defaultOf(namespace, scopeType));
    }

    public CompletableFuture<Void> resetLimitToDefault(Scope scope, String indicator) {
        if (scope.isDefaults()) {
            return CompletableFuture.failedFuture(new IllegalArgumentException("Cannot reset quota limit for defaults"));
        }

        return quotasDao.deleteOne(scope.getNamespace(), scope.getType(), scope.getIdentifier(), indicator);
    }

    public CompletableFuture<Void> deleteIndicator(String namespace, String scopeType, String indicator) {
        return quotasDao.delete(namespace, scopeType, indicator);
    }

    private CompletableFuture<List<QuotaLimit>> getLimits(Scope scope) {
        return quotasDao.findAllIndicators(scope.getNamespace(), scope.getType(), scope.getIdentifier())
            .thenApply(QuotaManager::coalesceLimits);
    }

    private static List<QuotaLimit> coalesceLimits(List<Quota> limits) {
        Set<String> scopeSpecificIndicators = limits.stream()
            .filter(quota -> !quota.isDefaultScopeId())
            .map(Quota::getIndicator)
            .collect(Collectors.toSet());

        return limits.stream()
            .filter(quota -> !(quota.isDefaultScopeId() && scopeSpecificIndicators.contains(quota.getIndicator())))
            .map(QuotaLimit::fromModel)
            .collect(Collectors.toList());
    }

    private List<QuotaValueWithLimit> mergeValuesWithLimits(Tuple2<List<QuotaValue>, List<QuotaLimit>> valueWithLimits) {
        final List<QuotaValue> values = valueWithLimits.get1();
        final List<QuotaLimit> limits = valueWithLimits.get2();

        Map<String, Long> fetchedValues = values.stream()
            .collect(Collectors.toMap(QuotaValue::getIndicator, QuotaValue::getValue));

        return limits.stream()
            .map(quotaLimit -> new QuotaValueWithLimit(
                quotaLimit.getIndicator(),
                fetchedValues.getOrDefault(quotaLimit.getIndicator(), 0L),
                quotaLimit.getLimit()))
            .collect(Collectors.toList());
    }

    private void register(QuotaFetcher fetcher) {
        String namespace = fetcher.getNamespace();
        delegatesByNamespace.computeIfAbsent(namespace, ignore -> new ArrayList<>()).add(fetcher);
        for (String indicator : fetcher.getRegisteredIndicators()) {
            @Nullable String maybeNamespace = namespaceByIndicator.get(indicator);
            if (maybeNamespace != null) {
                throw new IllegalStateException("Attempt to register a duplicate indicator '" + indicator + "' in " +
                        "namespaces '" + namespace + "' and '" + maybeNamespace + "'");
            }
            namespaceByIndicator.put(indicator, namespace);
        }
    }

    public Collection<String> getRegisteredNamespaces() {
        return delegatesByNamespace.keySet();
    }

    public Optional<String> getNamespaceByIndicator(String indicator) {
        return Optional.ofNullable(namespaceByIndicator.get(indicator));
    }
}
