package ru.yandex.solomon.quotas.manager.fetcher;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.quotas.manager.QuotaFetcher;
import ru.yandex.solomon.quotas.manager.QuotaValue;
import ru.yandex.solomon.quotas.manager.Scope;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AlertingQuotaFetcher implements QuotaFetcher {

    private static final String ALERT_COUNT = "alerts.count";
    private static final List<String> REGISTERED_INDICATORS = List.of(ALERT_COUNT);

    private final AlertingClient alertClient;

    public AlertingQuotaFetcher(AlertingClient alertClient) {
        this.alertClient = alertClient;
    }

    @Override
    public String getNamespace() {
        return "alerting";
    }

    @Override
    public List<String> getRegisteredIndicators() {
        return REGISTERED_INDICATORS;
    }

    @Override
    public CompletableFuture<List<QuotaValue>> fetch(Scope scope) {
        if (scope.getType().equals("project")) {
            String projectId = scope.getIdentifier();
            return alertClient.readProjectStats(TReadProjectStatsRequest.newBuilder()
                    .setProjectId(Strings.nullToEmpty(projectId))
                    .build())
                .thenCompose(response -> {
                    if (response.getRequestStatus() != ERequestStatusCode.OK) {
                        return failedFuture(GrpcApiException.of(response.getRequestStatus(), response.getStatusMessage()));
                    }

                    return completedFuture(List.of(new QuotaValue(ALERT_COUNT, response.getAlertsCount())));
                });
        }
        return completedFuture(List.of());
    }
}
