package ru.yandex.solomon.quotas.manager.fetcher;

import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.quotas.manager.QuotaFetcher;
import ru.yandex.solomon.quotas.manager.QuotaValue;
import ru.yandex.solomon.quotas.manager.Scope;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ManualAlertingQuotaFetcherStub implements QuotaFetcher {

    private static final String ALERT_COUNT = "alert.count";
    private static final String SUBALERT_COUNT = "subalert.count";
    private static final String CHANNEL_COUNT = "channel.count";

    private static final List<String> REGISTERED_INDICATORS = List.of(ALERT_COUNT, SUBALERT_COUNT, CHANNEL_COUNT);

    private static final Function<String, AtomicLong> DEFAULT_FILL = ignore -> new AtomicLong(0);

    private HashMap<String, AtomicLong> alertCount = new HashMap<>();
    private HashMap<String, AtomicLong> subalertCount = new HashMap<>();

    public AtomicLong getAlertCounter(String projectId) {
        return alertCount.computeIfAbsent(projectId, DEFAULT_FILL);
    }

    public AtomicLong getSubalertCounter(String projectId) {
        return subalertCount.computeIfAbsent(projectId, DEFAULT_FILL);
    }

    @Override
    public String getNamespace() {
        return "alerting";
    }

    @Override
    public List<String> getRegisteredIndicators() {
        return REGISTERED_INDICATORS;
    }

    @Override
    public CompletableFuture<List<QuotaValue>> fetch(Scope scope) {
        if (scope.getType().equals("project")) {
            String projectId = Objects.requireNonNull(scope.getIdentifier(), "scopeId");

            var result = List.of(
                new QuotaValue(ALERT_COUNT, getAlertCounter(projectId).get()),
                new QuotaValue(SUBALERT_COUNT, getSubalertCounter(projectId).get())
            );
            return completedFuture(result);
        }

        return completedFuture(List.of());
    }
}
