package ru.yandex.solomon.quotas.watcher;

import java.time.Duration;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalLong;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import org.apache.commons.lang3.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.core.db.dao.QuotasDao;
import ru.yandex.solomon.core.db.model.Quota;
import ru.yandex.solomon.quotas.watcher.pumpkin.PumpkinQuotas;
import ru.yandex.solomon.util.file.FileStorage;

/**
 * @author Ivan Tsybulin
 */
public class QuotaWatcher implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(QuotaWatcher.class);

    @VisibleForTesting
    static final String STATE_FILE = "quota_watcher.state";

    @Nullable
    private final FileStorage storage;
    private final QuotasDao quotasDao;
    private final String namespace;

    private volatile Map<Key, Long> quotas;
    private final ScheduledExecutorService timer;

    private static final long UPDATE_INTERVAL_MILLIS = Duration.ofMinutes(1).toMillis();
    private static final long UPDATE_INTERVAL_MAX_MILLIS = Duration.ofMinutes(10).toMillis();

    private volatile boolean done;
    private volatile ScheduledFuture<?> future;

    private Consumer<Void> onLoad;

    public QuotaWatcher(
        @Nullable FileStorage storage,
        QuotasDao quotasDao,
        String namespace,
        ScheduledExecutorService timer)
    {
        this.storage = storage;
        this.quotasDao = quotasDao;
        this.namespace = namespace;

        this.quotas = loadFromDisk();
        this.timer = timer;
        this.done = false;

        scheduleUpdate(UPDATE_INTERVAL_MILLIS, 0);
    }

    @VisibleForTesting
    public void registerOnLoad(Consumer<Void> onLoad) {
        this.onLoad = onLoad;
    }

    private long retryDelay(int retries) {
        long delayMillis = Math.round(ThreadLocalRandom.current().nextDouble(0.75, 1.25) * UPDATE_INTERVAL_MILLIS);
        delayMillis += (retries * retries) * UPDATE_INTERVAL_MILLIS;
        return Math.min(UPDATE_INTERVAL_MAX_MILLIS, delayMillis);
    }

    private void scheduleUpdate(long delayMillis, int retry) {
        if (done) {
            return;
        }
        future = timer.schedule(() -> loadFromYdbAndReschedule(retry), delayMillis, TimeUnit.MILLISECONDS);
    }

    private CompletableFuture<Map<Key, Long>> loadFromYdbAndReschedule(int retries) {
        return loadFromDb().whenComplete((loadedQuotas, e) -> {
                if (e != null) {
                    scheduleUpdate(retryDelay(retries), retries + 1);
                    logger.error("Failed to fetch quotas from db", e);
                } else {
                    quotas = loadedQuotas;
                    scheduleUpdate(retryDelay(0), 0);
                    logger.info("Fetched quotas from db, total " + loadedQuotas.size() + " records");
                }
                onLoad.accept(null);
            });
    }

    private Map<Key, Long> loadFromDisk() {
        if (storage == null) {
            return PumpkinQuotas.get(namespace);
        }

        try {
            var quotas = storage.loadValues(STATE_FILE, QuotaWatcher::deserialize);
            if (quotas == null) {
                throw new RuntimeException("file is not readable");
            }

            return quotas.stream()
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        } catch (Exception e) {
            logger.error("Error reading state file " + STATE_FILE + ", using pumpkin quotas!", e);
            return PumpkinQuotas.get(namespace);
        }
    }

    private void dumpToDisk(Map<Key, Long> quotas) {
        if (storage == null) {
            return;
        }

        try {
            storage.saveValues(STATE_FILE, quotas.entrySet(), QuotaWatcher::serialize);
        } catch (Exception e) {
            logger.error("Error while writing state file " + STATE_FILE, e);
        }
    }

    private CompletableFuture<Map<Key, Long>> loadFromDb() {
        return quotasDao.findAllByNamespace(namespace)
                .thenApply(quotas -> quotas.stream()
                .collect(Collectors.toMap(Key::fromRow, Quota::getLimit)))
                .whenComplete((quotasFromDb, e) -> {
                    if (e == null) {
                        dumpToDisk(quotasFromDb);
                    }
                });
    }

    public OptionalLong getLimit(String scopeType, String scopeId, String indicator) {
        final Map<Key, Long> quotas = this.quotas;

        Key key = Key.of(scopeType, scopeId, indicator);

        Long projectSpecificLimit = quotas.get(key);
        if (projectSpecificLimit != null) {
            return OptionalLong.of(projectSpecificLimit);
        }

        Key defaultsKey = Key.ofDefaults(scopeType, indicator);
        Long defaultsLimit = quotas.get(defaultsKey);

        return (defaultsLimit != null) ? OptionalLong.of(defaultsLimit) : OptionalLong.empty();
    }

    @VisibleForTesting
    static String serialize(Map.Entry<Key, Long> entry) {
        Key key = entry.getKey();
        long value = entry.getValue();
        return StringEscapeUtils.escapeJson(key.scopeType) + "\t" +
            StringEscapeUtils.escapeJson(Strings.nullToEmpty(key.scopeId)) + "\t" +
            StringEscapeUtils.escapeJson(key.indicator) + "\t" +
            value;
    }

    @VisibleForTesting
    static Map.Entry<Key, Long> deserialize(String line) {
        String[] tokens = line.split("\t", -1);
        Key key = new Key(
            StringEscapeUtils.unescapeJson(tokens[0]),
            Strings.emptyToNull(StringEscapeUtils.unescapeJson(tokens[1])),
            StringEscapeUtils.unescapeJson(tokens[2]));
        long value = Long.parseLong(tokens[3]);
        return Map.entry(key, value);
    }

    @Override
    public void close() {
        this.done = true;
        this.future.cancel(true);
    }

    public static class Key {
        final String scopeType;
        @Nullable
        final String scopeId;
        final String indicator;

        private Key(String scopeType, @Nullable String scopeId, String indicator) {
            this.scopeType = scopeType;
            this.scopeId = scopeId;
            this.indicator = indicator;
        }

        public static Key ofDefaults(String scopeType, String indicator) {
            return new Key(scopeType, null, indicator);
        }

        public static Key of(String scopeType, String scopeId, String indicator) {
            return new Key(scopeType, scopeId, indicator);
        }

        public static Key fromRow(Quota row) {
            return new Key(row.getScopeType(), row.getScopeId(), row.getIndicator());
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return scopeType.equals(key.scopeType) &&
                Objects.equals(scopeId, key.scopeId) &&
                indicator.equals(key.indicator);
        }

        @Override
        public int hashCode() {
            return Objects.hash(scopeType, scopeId, indicator);
        }

        @Override
        public String toString() {
            return "Key{" +
                "scopeType='" + scopeType + '\'' +
                ", scopeId='" + scopeId + '\'' +
                ", indicator='" + indicator + '\'' +
                '}';
        }
    }
}
