package ru.yandex.solomon.roles;

import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.acl.db.GroupMemberDao;
import ru.yandex.solomon.acl.db.model.GroupMember;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.staff.StaffClient;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Alexey Trushkin
 */
public class GroupMembersUpdateScheduler implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(GroupMembersUpdateScheduler.class);

    private final DistributedLock lock;
    private final Clock clock;
    private final long initialDelayMillis;
    private final long minReindexPeriod;
    private volatile long latestReindexTs;
    private volatile boolean closed;
    private final ScheduledExecutorService timer;
    private final PingActorRunner actor;
    private final AsyncMetrics metrics;
    private final GroupMemberDao groupMemberDao;
    private final StaffClient staffClient;

    public GroupMembersUpdateScheduler(
            GroupMemberDao groupMemberDao,
            StaffClient staffClient,
            DistributedLock lock,
            ExecutorService executor,
            ScheduledExecutorService timer,
            Clock clock,
            MetricRegistry registry,
            long intervalMillis)
    {
        this.groupMemberDao = groupMemberDao;
        this.staffClient = staffClient;
        this.lock = lock;
        this.clock = clock;
        this.initialDelayMillis = Duration.ofMinutes(1).toMillis();
        minReindexPeriod = intervalMillis / 2;
        this.timer = timer;
        this.metrics = new AsyncMetrics(registry, "staff.groupMembers.update");
        this.actor = PingActorRunner.newBuilder()
                .executor(executor)
                .timer(timer)
                .operation("Update staff group members")
                .pingInterval(Duration.ofMillis(intervalMillis))
                .backoffDelay(Duration.ofMinutes(1))
                .onPing(this::act)
                .build();
        acquireLock();
    }

    @VisibleForTesting
    public CompletableFuture<Void> updateMembers() {
        return groupMemberDao.getAll()
                .thenCompose(groupMembers -> {
                    var membersMap = groupMembers.stream().collect(Collectors.groupingBy(GroupMember::groupId));
                    var iterator = membersMap.entrySet().iterator();
                    AsyncActorBody body = () -> {
                        while (true) {
                            if (!iterator.hasNext()) {
                                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                            }

                            var entry = iterator.next();
                            return staffClient.getStaffGroupMembers(entry.getKey())
                                    .thenCompose(staffGroupMembers -> {
                                        var ourMembersMap = entry.getValue().stream().collect(Collectors.toMap(GroupMember::userId, o -> o));
                                        if (staffGroupMembers == null || staffGroupMembers.isEmpty()) {
                                            return groupMemberDao.delete(ourMembersMap.values());
                                        }
                                        var newMembers = new ArrayList<GroupMember>(staffGroupMembers.size());
                                        for (var staffGroupMember : staffGroupMembers) {
                                            if (ourMembersMap.remove(staffGroupMember.login) == null) {
                                                newMembers.add(new GroupMember(entry.getKey(), staffGroupMember.login));
                                            }
                                        }
                                        return groupMemberDao.delete(ourMembersMap.values())
                                                .thenCompose(unused -> groupMemberDao.create(newMembers));
                                    });
                        }
                    };
                    var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 10);
                    return runner.start();
                });
    }

    public CompletableFuture<Void> act(int attempt) {
        if (closed) {
            return completedFuture(null);
        }

        if (!lock.isLockedByMe()) {
            return completedFuture(null);
        }
        long reindexTs = clock.millis();
        if (minReindexPeriod > reindexTs - latestReindexTs) {
            return completedFuture(null);
        }

        var future = updateMembers()
                .thenAccept(ignore -> latestReindexTs = reindexTs);
        metrics.forFuture(future);
        return future;
    }

    public void schedule() {
        long delay = DurationUtils.randomize(initialDelayMillis);
        timer.schedule(actor::forcePing, delay, TimeUnit.MILLISECONDS);
    }

    private void acquireLock() {
        if (closed) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Acquire groupMemberUpdater lock, seqNo {}", seqNo);
                schedule();
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Loose groupMemberUpdater lock by reason: {}", reason);
                acquireLock();
            }
        }, 5, TimeUnit.MINUTES);
    }

    @Override
    public void close() {
        closed = true;
        actor.close();
    }
}
