package ru.yandex.solomon.roles;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import ru.yandex.solomon.acl.db.ProjectAclEntryDao;
import ru.yandex.solomon.acl.db.model.AclUidType;
import ru.yandex.solomon.acl.db.model.ProjectAclEntry;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.roles.idm.IdmException;
import ru.yandex.solomon.roles.idm.dto.IdmAddRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmRemoveRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRoleTreeResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;
import ru.yandex.solomon.roles.idm.dto.RoleDto;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Alexey Trushkin
 */
public class ProjectRoleManager implements RoleManager {

    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(3)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final ProjectAclEntryDao dao;
    private final SolomonConfHolder confHolder;
    private final ProjectsDao projectsDao;

    public ProjectRoleManager(ProjectAclEntryDao dao, SolomonConfHolder confHolder, ProjectsDao projectsDao) {
        this.dao = dao;
        this.confHolder = confHolder;
        this.projectsDao = projectsDao;
    }

    @Override
    public boolean accepts(RoleDto role) {
        return role.isProject();
    }

    @Override
    public CompletableFuture<IdmResponseDto.ResultData> addRole(IdmAddRoleDto idmAddRoleDto) {
        var projectId = idmAddRoleDto.project;
        if (confHolder.getConfOrThrow().getProject(projectId) == null) {
            return projectsDao.findById(projectId).thenCompose(projectOptional -> {
                if (projectOptional.isEmpty()) {
                    return CompletableFuture.failedFuture(new IdmException("Hasn't project " + projectId, true));
                }
                return addRole(idmAddRoleDto, projectId);
            });
        }
        return addRole(idmAddRoleDto, projectId);
    }

    private CompletableFuture<IdmResponseDto.ResultData> addRole(IdmAddRoleDto idmAddRoleDto, String projectId) {
        var uid = idmAddRoleDto.getUid();
        var role = idmAddRoleDto.role.roleId;
        return RetryCompletableFuture.runWithRetries(() -> addRole(idmAddRoleDto, projectId, uid, role), RETRY_CONFIG);
    }

    private CompletableFuture<IdmResponseDto.ResultData> addRole(
            IdmAddRoleDto idmAddRoleDto,
            String projectId,
            String uid,
            String role)
    {
        return dao.find(projectId, uid, idmAddRoleDto.aclUidType)
                .thenCompose(entryOptional -> {
                    var entry = prepareEntry(entryOptional, projectId, idmAddRoleDto.aclUidType, uid, role);
                    if (entryOptional.isPresent()) {
                        return dao.update(entry);
                    } else {
                        return createNewEntry(entry);
                    }
                })
                .thenApply(unused -> IdmResponseDto.ResultData.of(projectId));
    }

    @Override
    public CompletableFuture<Void> removeRole(IdmRemoveRoleDto idmRemoveRoleDto) {
        var projectId = idmRemoveRoleDto.project;
        var uid = idmRemoveRoleDto.getUid();
        var role = idmRemoveRoleDto.role.roleId;
        return RetryCompletableFuture.runWithRetries(() -> removeRole(idmRemoveRoleDto, projectId, uid, role), RETRY_CONFIG);
    }

    private CompletableFuture<Void> removeRole(
            IdmRemoveRoleDto idmRemoveRoleDto,
            String projectId,
            String uid,
            String role)
    {
        return dao.find(projectId, uid, idmRemoveRoleDto.aclUidType)
                .thenCompose(entryOptional -> {
                    if (entryOptional.isPresent()) {
                        var entry = entryOptional.get();
                        entry.getRoles().remove(role);
                        if (entry.getRoles().isEmpty()) {
                            return deleteEntry(entry);
                        } else {
                            return dao.update(entry);
                        }
                    } else {
                        return CompletableFuture.completedFuture(null);
                    }
                });
    }

    @Override
    public CompletableFuture<IdmRoleTreeResponseDto.RoleSubTree> getRoleSubTree() {
        var tree = IdmRoleTreeResponseDto.RoleSubTree.of(
                RoleDto.Type.PROJECT, "Роль в проекте", "Project role");
        var field = IdmRoleTreeResponseDto.Field.of("project", "Проект", "Project", "charfield", true);
        tree.fields = List.of(field);
        tree.addRole(Role.PROJECT_ADMIN, "Администратор проекта", "Project admin");
        tree.addRole(Role.VIEWER, "Читатель", "Viewer");
        tree.addRole(Role.EDITOR, "Редактор конфигурации", "Config editor");
        tree.addRole(Role.PUSHER, "Писатель метрик", "Data writer");
        tree.addRole(Role.MUTER, "Управляющий мьютами", "Muter");
        tree.addRole(Role.JNS_SENDER, "Отправитель JNS", "JNS sender");
        return CompletableFuture.completedFuture(tree);
    }

    @Override
    public CompletableFuture<List<IdmRolesPageResponseDto.Role>> getRoles() {
        return dao.getAll()
                .thenApply(entries -> {
                    List<IdmRolesPageResponseDto.Role> roles = new ArrayList<>(entries.size());
                    for (var entry : entries) {
                        roles.addAll(IdmRolesPageResponseDto.Role.createRoles(entry));
                    }
                    return roles;
                });
    }

    private CompletableFuture<Void> deleteEntry(ProjectAclEntry entry) {
        return dao.delete(entry).thenAccept(result -> validateDaoResult(result, entry));
    }

    private CompletableFuture<Void> createNewEntry(ProjectAclEntry entry) {
        return dao.create(entry).thenAccept(result -> validateDaoResult(result, entry));
    }

    private ProjectAclEntry prepareEntry(
            Optional<ProjectAclEntry> optional,
            String projectId,
            AclUidType type,
            String uid,
            String role)
    {
        if (optional.isPresent()) {
            ProjectAclEntry entry = optional.get();
            entry.getRoles().add(role);
            return entry;
        }
        return new ProjectAclEntry(projectId, uid, type, Set.of(role), 0);
    }

    private void validateDaoResult(Boolean result, ProjectAclEntry entry) {
        if (!result) {
            String message = String.format(
                    "ProjectAclEntry (%s) with version %s is out of date",
                    entry.getCompositeId(),
                    entry.getVersion()
            );
            throw new ConflictException(message);
        }
    }

}
