package ru.yandex.solomon.roles;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

import com.google.common.base.Throwables;

import ru.yandex.idm.IdmClient;
import ru.yandex.idm.dto.RoleRequestDto;
import ru.yandex.idm.http.HttpIdmClient;
import ru.yandex.solomon.auth.roles.Role;
import ru.yandex.solomon.config.protobuf.TIdmConfig;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.model.Acl;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.roles.idm.IdmException;
import ru.yandex.solomon.roles.idm.dto.IdmAddRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmRemoveRoleDto;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Alexey Trushkin
 */
public class RoleAclSynchronizer implements RoleChangeListener {

    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(3)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private final ProjectsDao projectsDao;
    private final TIdmConfig config;
    private final IdmClient idmClient;

    public RoleAclSynchronizer(
            ProjectsDao projectsDao,
            TIdmConfig config,
            IdmClient idmClient)
    {
        this.projectsDao = projectsDao;
        this.config = config;
        this.idmClient = idmClient;
    }

    @Override
    public CompletableFuture<Void> roleAdded(IdmAddRoleDto dto) {
        if (dto.isGroup() || !dto.role.isProject()) {
            return CompletableFuture.completedFuture(null);
        }
        return RetryCompletableFuture.runWithRetries(() -> roleAddedInner(dto), RETRY_CONFIG);
    }

    private CompletableFuture<Void> roleAddedInner(IdmAddRoleDto dto) {
        return projectsDao.findById(dto.project)
                .thenCompose(projectOptional -> {
                    if (projectOptional.isEmpty()) {
                        return CompletableFuture.failedFuture(new IdmException("Hasn't project " + dto.project, true));
                    }
                    Project project = projectOptional.get();
                    var acl = project.getAcl();
                    var login = dto.login;
                    if (dto.isTvm()) {
                        login = "tvm-" + login;
                    }
                    if (Role.PROJECT_ADMIN.name().equals(dto.role.roleId)) {
                        if (!acl.getCanUpdate().contains(login) || !acl.getCanDelete().contains(login)) {
                            var update = new HashSet<>(acl.getCanUpdate());
                            update.add(login);
                            var delete = new HashSet<>(acl.getCanDelete());
                            delete.add(login);
                            return updateProject(project.toBuilder()
                                    .setAcl(Acl.of(acl.getCanRead(), update, delete, acl.getCanWrite()))
                                    .build());
                        }
                    } else if (Role.VIEWER.name().equals(dto.role.roleId)) {
                        if (!acl.getCanRead().contains(login)) {
                            var read = new HashSet<>(acl.getCanRead());
                            read.add(login);
                            return updateProject(project.toBuilder()
                                    .setAcl(Acl.of(read, acl.getCanUpdate(), acl.getCanDelete(), acl.getCanWrite()))
                                    .build());
                        }
                    } else if (Role.PUSHER.name().equals(dto.role.roleId)) {
                        if (!acl.getCanWrite().contains(login)) {
                            var write = new HashSet<>(acl.getCanWrite());
                            write.add(login);
                            return updateProject(project.toBuilder()
                                    .setAcl(Acl.of(acl.getCanRead(), acl.getCanUpdate(), acl.getCanDelete(), write))
                                    .build());
                        }
                    }
                    return CompletableFuture.completedFuture(null);
                });
    }

    private CompletionStage<Void> updateProject(Project project) {
        return projectsDao.partialUpdate(project, false, false, true)
                .thenApply(updatedProject -> {
                    if (updatedProject.isPresent()) {
                        return updatedProject.get();
                    }
                    throw projectIsOutOfDate(project.getId(), project.getVersion());
                })
                .thenCompose(unused -> CompletableFuture.completedFuture(null));
    }

    @Override
    public CompletableFuture<Void> roleRemoved(IdmRemoveRoleDto dto) {
        if (dto.isGroup() || !dto.role.isProject()) {
            return CompletableFuture.completedFuture(null);
        }
        return RetryCompletableFuture.runWithRetries(() -> roleRemovedInner(dto), RETRY_CONFIG);
    }

    private CompletableFuture<Void> roleRemovedInner(IdmRemoveRoleDto dto) {
        return projectsDao.findById(dto.project)
                .thenCompose(projectOptional -> {
                    if (projectOptional.isEmpty()) {
                        return CompletableFuture.completedFuture(null);
                    }
                    Project project = projectOptional.get();
                    var acl = project.getAcl();
                    var login = dto.login;
                    if (dto.isTvm()) {
                        login = "tvm-" + login;
                    }
                    if (Role.PROJECT_ADMIN.name().equals(dto.role.roleId)) {
                        if (acl.getCanUpdate().contains(login) || acl.getCanDelete().contains(login)) {
                            var update = new HashSet<>(acl.getCanUpdate());
                            update.remove(login);
                            var delete = new HashSet<>(acl.getCanDelete());
                            delete.remove(login);
                            return updateProject(project.toBuilder()
                                    .setAcl(Acl.of(acl.getCanRead(), update, delete, acl.getCanWrite()))
                                    .build());
                        }
                    } else if (Role.VIEWER.name().equals(dto.role.roleId)) {
                        if (acl.getCanRead().contains(login)) {
                            var read = new HashSet<>(acl.getCanRead());
                            read.remove(login);
                            return updateProject(project.toBuilder()
                                    .setAcl(Acl.of(read, acl.getCanUpdate(), acl.getCanDelete(), acl.getCanWrite()))
                                    .build());
                        }
                    } else if (Role.PUSHER.name().equals(dto.role.roleId)) {
                        if (acl.getCanWrite().contains(login)) {
                            var write = new HashSet<>(acl.getCanWrite());
                            write.remove(login);
                            return updateProject(project.toBuilder()
                                    .setAcl(Acl.of(acl.getCanRead(), acl.getCanUpdate(), acl.getCanDelete(), write))
                                    .build());
                        }
                    }
                    return CompletableFuture.completedFuture(null);
                });
    }

    public CompletableFuture<Void> changed(Project oldProject, Project updatedProject) {
        var oldAcl = oldProject.getAcl();
        var newAcl = updatedProject.getAcl();
        if (oldAcl.equals(newAcl)) {
            return CompletableFuture.completedFuture(null);
        }
        Set<String> adminOld = new HashSet<>(oldAcl.getCanUpdate());
        adminOld.addAll(oldAcl.getCanDelete());
        Set<String> adminNew = new HashSet<>(newAcl.getCanUpdate());
        adminNew.addAll(newAcl.getCanDelete());
        return syncAcl(adminOld, adminNew, updatedProject.getId(), Role.PROJECT_ADMIN)
                .thenCompose(unused -> syncAcl(oldAcl.getCanRead(), newAcl.getCanRead(), updatedProject.getId(), Role.VIEWER))
                .thenCompose(unused3 -> syncAcl(oldAcl.getCanWrite(), newAcl.getCanWrite(), updatedProject.getId(), Role.PUSHER));
    }

    private CompletableFuture<Void> syncAcl(Set<String> oldSet, Set<String> newSet, String projectId, Role... roles) {
        Set<String> toAdd = new HashSet<>(newSet);
        List<String> toDelete = new ArrayList<>();
        for (String login : oldSet) {
            if (!toAdd.contains(login)) {
                toDelete.add(login);
            } else {
                toAdd.remove(login);
            }
        }
        if (toAdd.isEmpty() && toDelete.isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        List<RoleRequestDto> addRequests = new ArrayList<>(toAdd.size());
        List<RoleRequestDto> deleteRequests = new ArrayList<>(toDelete.size());
        for (Role role : roles) {
            for (String login : toAdd) {
                var tvm = login.startsWith("tvm-");
                if (tvm) {
                    login = login.substring(4);
                }
                addRequests.add(RoleRequestDto.newProjectRoleRequest(login, projectId, role.name(), config.getSystemName(), tvm));
            }
            for (String login : toDelete) {
                var tvm = login.startsWith("tvm-");
                if (tvm) {
                    login = login.substring(4);
                }
                deleteRequests.add(RoleRequestDto.newProjectRoleRequest(login, projectId, role.name(), config.getSystemName(), tvm));
            }
        }
        return new ProjectRoleRequester(addRequests).start()
                .thenCompose(unused -> new ProjectRoleDeleter(deleteRequests).start());
    }

    private class ProjectRoleRequester {

        private final Iterator<RoleRequestDto> iterator;
        private final CompletableFuture<Void> result = new CompletableFuture<>();

        public ProjectRoleRequester(List<RoleRequestDto> addRequests) {
            iterator = addRequests.iterator();
        }

        public CompletableFuture<Void> start() {
            if (!iterator.hasNext()) {
                result.complete(null);
                return result;
            }
            requestNext();
            return result;
        }

        private void requestNext() {
            RoleRequestDto next = iterator.next();
            try {
                idmClient.requestRole(next)
                        .whenComplete(this::onResponse);
            } catch (Exception e) {
                result.completeExceptionally(e);
            }
        }

        private void onResponse(Void unused, Throwable e) {
            // skip fired and failed by client
            if (e != null
                    && !(Throwables.getRootCause(e) instanceof HttpIdmClient.FiredError)
                    && !(Throwables.getRootCause(e) instanceof HttpIdmClient.ClientError))
            {
                result.completeExceptionally(e);
                return;
            }

            if (iterator.hasNext()) {
                requestNext();
            } else {
                result.complete(null);
            }
        }
    }

    private class ProjectRoleDeleter {

        private final Iterator<RoleRequestDto> iterator;
        private final CompletableFuture<Void> result = new CompletableFuture<>();

        public ProjectRoleDeleter(List<RoleRequestDto> addRequests) {
            iterator = addRequests.iterator();
        }

        public CompletableFuture<Void> start() {
            if (!iterator.hasNext()) {
                result.complete(null);
                return result;
            }
            requestNext();
            return result;
        }

        private void requestNext() {
            RoleRequestDto next = iterator.next();
            try {
                idmClient.deleteRole(next)
                        .whenComplete(this::onResponse);
            } catch (Exception e) {
                result.completeExceptionally(e);
            }
        }

        private void onResponse(Void unused, Throwable e) {
            if (e != null) {
                result.completeExceptionally(e);
                return;
            }

            if (iterator.hasNext()) {
                requestNext();
            } else {
                result.complete(null);
            }
        }
    }


    private static ConflictException projectIsOutOfDate(String projectId, int version) {
        String message = String.format(
                "project %s with version %s is out of date",
                projectId,
                version
        );
        return new ConflictException(message);
    }
}
