package ru.yandex.solomon.roles;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfListener;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Alexey Trushkin
 */
public class RoleAclSynchronizerListener implements SolomonConfListener, AutoCloseable {

    private static final Logger logger = LoggerFactory.getLogger(RoleAclSynchronizerListener.class);
    private final DistributedLock lock;
    private final RoleAclSynchronizer roleAclSynchronizer;
    private final Map<String, Project> knownProjects;
    private volatile boolean closed;
    private final ActorWithFutureRunner actor;

    private volatile SolomonConfWithContext conf;

    public RoleAclSynchronizerListener(
            ExecutorService executor,
            DistributedLock lock,
            RoleAclSynchronizer roleAclSynchronizer)
    {
        this.lock = lock;
        this.roleAclSynchronizer = roleAclSynchronizer;
        knownProjects = new HashMap<>();
        this.actor = new ActorWithFutureRunner(this::sync, executor);
        acquireLock();
    }

    @Override
    public void onConfigurationLoad(SolomonConfWithContext conf) {
        this.conf = conf;
        if (closed) {
            return;
        }
        actor.schedule();
    }

    private CompletableFuture<?> sync() {
        if (!lock.isLockedByMe()) {
            return CompletableFuture.completedFuture(null);
        }
        try {
            var projectIds = new HashSet<>(conf.projects());
            if (projectIds.isEmpty()) {
                return completedFuture(null);
            }
            if (knownProjects.isEmpty()) {
                // initial state
                for (String projectId : projectIds) {
                    knownProjects.put(projectId, conf.getProject(projectId));
                }
                return CompletableFuture.completedFuture(null);
            }

            List<ProjectDiff> projectDiffs = new ArrayList<>(projectIds.size());
            for (String projectId : projectIds) {
                var newState = conf.getProject(projectId);
                var oldState = knownProjects.get(projectId);
                if (oldState != null) {
                    var oldAcl = oldState.getAcl();
                    var newAcl = newState.getAcl();
                    if (oldAcl.equals(newAcl)) {
                        continue;
                    }
                    projectDiffs.add(new ProjectDiff(oldState, newState));
                }
            }

            return new Updater(projectDiffs).start()
                    .handle((unused, throwable) -> {
                        if (throwable != null) {
                            logger.error("Error while update roles projects.", throwable);
                        } else {
                            knownProjects.clear();
                            for (String projectId : projectIds) {
                                knownProjects.put(projectId, conf.getProject(projectId));
                            }
                        }
                        return unused;
                    });
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private void acquireLock() {
        if (closed) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Acquire RoleAclSynchronizerListener lock, seqNo {}", seqNo);
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Loose RoleAclSynchronizerListener lock by reason: {}", reason);
                acquireLock();
            }
        }, 5, TimeUnit.MINUTES);
    }

    @Override
    public void close() {
        closed = true;
    }

    private class Updater {

        private final Iterator<ProjectDiff> iterator;
        private final CompletableFuture<Void> result = new CompletableFuture<>();

        public Updater(List<ProjectDiff> projectDiffs) {
            iterator = projectDiffs.iterator();
        }

        public CompletableFuture<Void> start() {
            if (!iterator.hasNext()) {
                result.complete(null);
                return result;
            }
            requestNext();
            return result;
        }

        private void requestNext() {
            var next = iterator.next();
            roleAclSynchronizer.changed(next.oldState, next.newState)
                    .whenComplete(this::onResponse);
        }

        private void onResponse(Void unused, Throwable e) {
            if (e != null) {
                result.completeExceptionally(e);
                return;
            }

            if (iterator.hasNext()) {
                requestNext();
            } else {
                result.complete(null);
            }
        }
    }

    private record ProjectDiff(Project oldState, Project newState) {
    }
}
