package ru.yandex.solomon.roles;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.roles.idm.IdmException;
import ru.yandex.solomon.roles.idm.dto.IdmAddRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmRemoveRoleDto;
import ru.yandex.solomon.roles.idm.dto.IdmResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRoleTreeResponseDto;
import ru.yandex.solomon.roles.idm.dto.IdmRolesPageResponseDto;
import ru.yandex.solomon.roles.idm.dto.RoleDto;

/**
 * @author Alexey Trushkin
 */
public class RoleServiceImpl implements RoleService {

    private final List<RoleManager> roleManagers;
    private final List<RoleChangeListener> listeners;

    public RoleServiceImpl(List<RoleManager> roleManagers, List<RoleChangeListener> listeners) {
        this.roleManagers = roleManagers;
        this.listeners = listeners;
    }

    @Override
    public CompletableFuture<IdmResponseDto.ResultData> addRole(IdmAddRoleDto idmAddRoleDto) {
        RoleManager roleManager = getRoleManager(idmAddRoleDto.role);
        return roleManager.addRole(idmAddRoleDto)
                .thenCompose(data -> {
                    List<CompletableFuture<Void>> futures = new ArrayList<>(listeners.size());
                    for (var listener : listeners) {
                        futures.add(listener.roleAdded(idmAddRoleDto));
                    }
                    return CompletableFutures.allOfVoid(futures)
                            .thenApply(unused -> data);
                });
    }

    @Override
    public CompletableFuture<Void> removeRole(IdmRemoveRoleDto idmRemoveRoleDto) {
        RoleManager roleManager = getRoleManager(idmRemoveRoleDto.role);
        return roleManager.removeRole(idmRemoveRoleDto)
                .thenCompose(aVoid -> {
                    List<CompletableFuture<Void>> futures = new ArrayList<>(listeners.size());
                    for (var listener : listeners) {
                        futures.add(listener.roleRemoved(idmRemoveRoleDto));
                    }
                    return CompletableFutures.allOfVoid(futures);
                });
    }

    @Override
    public CompletableFuture<IdmRoleTreeResponseDto.RoleTree> getRoleTree() {
        List<CompletableFuture<IdmRoleTreeResponseDto.RoleSubTree>> futures = new ArrayList<>(roleManagers.size());
        for (var roleManager : roleManagers) {
            futures.add(roleManager.getRoleSubTree());
        }
        return CompletableFutures.allOf(futures)
                .thenApply(roleSubTrees -> {
                    IdmRoleTreeResponseDto.RoleTree tree = IdmRoleTreeResponseDto.RoleTree.of("type", "Тип роли", "Role type");
                    for (var subTree : roleSubTrees) {
                        tree.values.put(subTree.type.name().toLowerCase(), subTree);
                    }
                    return tree;
                });
    }

    @Override
    public CompletableFuture<List<IdmRolesPageResponseDto.Role>> getRoles() {
        List<CompletableFuture<List<IdmRolesPageResponseDto.Role>>> futures = new ArrayList<>(roleManagers.size());
        for (var roleManager : roleManagers) {
            futures.add(roleManager.getRoles());
        }
        return CompletableFutures.allOf(futures)
                .thenApply(roles -> roles.stream()
                        .flatMap(Collection::stream)
                        .collect(Collectors.toList()));
    }

    private RoleManager getRoleManager(RoleDto role) {
        for (var roleManager : roleManagers) {
            if (roleManager.accepts(role)) {
                return roleManager;
            }
        }
        throw new IdmException("Hasn't role manager for " + role, true);
    }
}
